(ns lucid.legacy.analyzer.ast
  "Utilities for AST walking/updating"
  (:require [lucid.legacy.analyzer.utils :refer [into! rseqv]]))

(defn cycling
  ""
  [& fns*]
  (let [fns (cycle fns*)]
    (fn [ast]
      (loop [[f & fns] fns ast ast res (zipmap fns* (repeat nil))]
        (let [ast* (f ast)]
          (if (= ast* (res f))
            ast
            (recur fns ast* (assoc res f ast*))))))))

(defn children*
  ""
  [{:keys [children] :as ast}]
  (when children
    (mapv #(find ast %) children)))

(defn children
  ""
  [ast]
  (persistent!
   (reduce (fn [acc [_ c]] ((if (vector? c) into! conj!) acc c))
           (transient []) (children* ast))))

(defmulti -update-children
  ""   (fn [ast f] (:op ast)))
(defmulti -update-children-r
  "" (fn [ast f] (:op ast)))

(defmethod -update-children :default
  [ast f]
  (persistent!
   (reduce (fn [ast [k v]]
             (assoc! ast k (if (vector? v) (mapv f v) (f v))))
           (transient ast)
           (children* ast))))

(defmethod -update-children-r :default
  [ast f]
  (persistent!
   (reduce (fn [ast [k v]]
             (assoc! ast k (if (vector? v) (rseqv (mapv f (rseq v))) (f v))))
           (transient ast)
           (rseq (children* ast)))))

(defn update-children
  ""
  ([ast f] (update-children ast f false))
  ([ast f reversed?]
     (if (:children ast)
       (if reversed?
         (-update-children-r ast f)
         (-update-children   ast f))
       ast)))

(defn walk
  ""
  ([ast pre post]
     (walk ast pre post false))
  ([ast pre post reversed?]
     (let [walk #(walk % pre post reversed?)]
       (post (update-children (pre ast) walk reversed?)))))

(defn prewalk
  ""
  [ast f]
  (let [walk #(prewalk % f)]
    (update-children (f ast) walk)))

(defn postwalk
  ""
  ([ast f]
     (postwalk ast f false))
  ([ast f reversed?]
     (let [walk #(postwalk % f reversed?)]
       (f (update-children ast walk reversed?)))))

(defn nodes
  ""
  [ast]
  (lazy-seq
   (cons ast (mapcat nodes (children ast)))))

(defn ast->eav
  ""
  [ast]
  (let [children (set (:children ast))]
    (mapcat (fn [[k v]]
              (if (children k)
                (if (map? v)
                  (into [[ast k v]] (ast->eav v))
                  (mapcat (fn [v] (into [[ast k v]] (ast->eav v))) v))
                [[ast k v]])) ast)))
