(ns hara.reflect.element.method
  (:require [hara.reflect.common :refer :all]
            [hara.reflect.hierarchy :as hierarchy]
            [hara.reflect.types.element :refer :all]
            [hara.reflect.element.common :refer :all]
            [hara.reflect.pretty.classes :refer [class-convert]]))

(defn invoke-static-method
  "invoke function for a static method
 
   (-> (query/query-class clojure.java.api.Clojure [\"read\" :#])
       (method/invoke-static-method [\"{:a 1}\"]))
   => {:a 1}"
  {:added "2.1"}
  ([ele]
     (try (.invoke ^java.lang.reflect.Method (:delegate ele) nil (object-array []))
          (catch IllegalArgumentException e
            (throw-arg-exception ele []))))
  ([ele args]
     (.invoke ^java.lang.reflect.Method (:delegate ele) nil (object-array (box-args ele args)))))

(defn invoke-instance-method
  "invoke function for an instance method
 
   (-> (query/query-class String [\"charAt\" :#])
       (method/invoke-instance-method [\"0123\" 1]))
   => \1"
  {:added "2.1"}
  [ele args]
  (let [bargs (box-args ele args)]
    (.invoke ^java.lang.reflect.Method (:delegate ele) (first bargs) (object-array (rest bargs)))))

(defmethod invoke-element :method
  ([ele]
     (if (:static ele)
       (invoke-static-method ele)
       (throw-arg-exception ele [])))
  ([ele & args]
     (if (:static ele)
       (invoke-static-method ele args)
       (invoke-instance-method ele args))))

(defn to-static-method
  "creates the parameters for a static method
 
   (-> (query/query-class clojure.java.api.Clojure [\"read\" :#])
       :delegate
       (method/to-static-method {}))
   => {:params [String]
       :origins [clojure.java.api.Clojure]}"
  {:added "2.1"}
  [^java.lang.reflect.Method obj body]
  (-> body
      (assoc :params (vec (seq (.getParameterTypes obj))))
      (assoc :origins (list (.getDeclaringClass obj)))))

(defn to-instance-method
  "creates the parameters for an instance method
 
   (-> (query/query-class String [\"charAt\" :#])
       :delegate
       (method/to-instance-method {:container String}))
   => {:container String
       :params [String Integer/TYPE]
       :origins [CharSequence String]}"
  {:added "2.1"}
  [^java.lang.reflect.Method obj body]
  (-> body
      (assoc :params (vec (cons (:container body) (seq (.getParameterTypes obj)))))
      (assoc :origins (hierarchy/origins obj))))

(defn to-pre-element
  "creates the parameters for methods
 
   (-> (query/query-class String [\"charAt\" :#])
       :delegate
       (method/to-pre-element))
   => (contains {:name \"charAt\"
                 :tag :method
                 :container String
                :modifiers #{:instance :method :public}
                 :static false
                 :delegate java.lang.reflect.Method
                 :params [String Integer/TYPE]
                 :origins [CharSequence String]})"
  {:added "2.1"}
  [obj]
  (let [body (seed :method obj)
        body (if (:static body)
               (to-static-method obj body)
               (to-instance-method obj body))]
    body))

(defmethod to-element java.lang.reflect.Method
  [^java.lang.reflect.Method obj]
  (let [body (-> (to-pre-element obj)
                 (assoc :type (.getReturnType obj)))]
    (element body)))

(defmethod format-element :method [ele]
  (format-element-method ele))

(defmethod element-params :method [ele]
  (list (element-params-method ele)))
