(ns hara.reflect.types.element
  (:require [clojure.walk :as walk]))

(defmulti invoke-element
  "base method for extending `invoke` for all element types"
  {:added "2.1"}
  (fn [x & args] (:tag x)))

(defmulti to-element
  "base method for extending creating an element from java.reflect objects"
  {:added "2.1"}
  type)

(defmulti element-params
  "base method for extending `:params` entry for all element types"
  {:added "2.1"}
  :tag)

(defmulti format-element
  "base method for extending `toString` entry for all element types"
  {:added "2.1"}
  :tag)

(defn make-invoke-element-form
  "helper function for making invoke form
   
   (make-invoke-element-form '[a1 a2])
   => '(invoke [ele a1 a2] (invoke-element ele a1 a2))"
  {:added "2.1"}
  [args]
  (walk/postwalk
   (fn [x]
     (cond (and (list? x)
                (= 'invoke-element (first x)))
           (concat x args)

           (vector? x)
           (vec (concat x args))
           :else x))
   '(invoke [ele]
            (invoke-element ele))))

(defmacro init-element-type
  "helper function for making `Element` type
 
   (macroexpand-1
   '(init-element-type 4))"
  {:added "2.1"}
  [n]
  (concat
   '(deftype Element [body]
      java.lang.Object
      (toString [ele]
        (format-element ele))

      clojure.lang.ILookup
      (valAt [ele k]
        (if (or (nil? k)
                (= k :all))
          body
          (get body k)))

      clojure.lang.IFn
      (applyTo [ele args]
        (clojure.lang.AFn/applyToHelper ele args)))
   (map make-invoke-element-form
        (for [l (range n)]
          (vec (for [x (range l)]
                 (symbol (str "arg" x))))))))

(init-element-type 20)

(defn element
  "creates a element from a map
 
   (element {})
   => hara.reflect.types.element.Element"
  {:added "2.1"}
  [body]
  (Element. body))

(defn element?
  "checker for the element type
 
   (element? (element {}))
   => true"
  {:added "2.1"}
  [x]
  (instance? Element x))

(defmethod print-method Element
  [v ^java.io.Writer w]
  (.write w (str v)))
