(ns hara.image.base
  (:require [hara.protocol.image :as image]
            [hara.image.base
             [common :as common]
             [display :as display]
             [size :as size]
             [util :as util]]
            [hara.image.awt
             [common :as awt]
             [io :as io]]
            [hara.image.model :as model])
  (:import clojure.lang.IPersistentMap
           clojure.lang.PersistentArrayMap))

(defrecord Image [model size data]
  Object
  (toString [_]
    (str "#image.base" [(image/-width size)
                 (image/-height size)]
         {:model (:label model)}))

  image/IRepresentation
  (-channels [_] ((-> model :channel :fn) data))
  (-size     [_] size)
  (-model    [_] model)
  (-data     [_] data)
  (-subimage [img x y w h]
    (common/subimage img x y w h))

  image/ITransfer
  (-to-byte-gray [image]
    (-> (common/convert-base image (model/model :byte-gray))
        :data))
  (-to-int-argb  [image]
    (-> (common/convert-base image (model/model :int-argb))
        :data))
  (-write [image sink opts]
    (io/write (awt/image image) sink opts)))

(defmethod print-method Image
  [v w]
  (.write w (str v)))

(defn image
  "creates a new image record for holding data
 
   (image {:size [100 100]
           :model :ushort-555-rgb
           :data nil})
   ;;=> #img[100 100]{:format :ushort-555-rgb}
   "
  {:added "2.8"}
  ([{:keys [model size data] :as m}]
   (-> (map->Image m)
       (update-in [:size] size/size->map)
       (update-in [:model] model/model)
       (assoc :type IPersistentMap))))

(defmethod image/-read Image
  [source model _]
  (-> (io/read source)
      (common/convert model)
      image))

(defmethod image/-read :base
  [source model _]
  (image/-read source model Image))

(defmethod image/-image Image
  [size model data _]
  (image {:size size
          :model model
          :data data}))

(defmethod image/-image :base
  [size model data _]
  (image/-image size model data Image))

(defmethod image/-image IPersistentMap
  [size model data _]
  (image/-image size model data Image))

(defmethod image/-blank Image
  [size model _]
  (let [model (model/model model)
        channels (common/create-channels model (size/length size))
        data ((-> model :channel :inv) channels)]
    (image {:size size
            :model model
            :data data})))

(defmethod image/-blank :base
  [size model _]
  (image/-blank size model Image))

(defmethod image/-blank IPersistentMap
  [size model _]
  (image/-blank size model Image))

(defmethod image/-display :base
  [img opts _]
  (display/display img opts))

(defmethod image/-display Image
  [img opts _]
  (display/display img opts))

(defmethod image/-display-class :base
  [_]
  #{Image})

(defmethod image/-display-class Image
  [_]
  #{Image})

(defmethod image/-display-class :base
  [_]
  #{Image})
