(ns hara.image
  (:require [hara.image base awt]
            [hara.image.base.common :as common]
            [hara.image.model :as model]
            [hara.protocol.image :as image])
  (:import java.awt.image.BufferedImage)
  (:refer-clojure :exclude [read]))

(def ^:dynamic *default-type* BufferedImage)

(def ^:dynamic *default-model* (model/model :int-argb))

(def ^:dynamic *default-view* :base)

(defn image?
  "checks whether object is an image
 
   (image? (image/read \"resources/data/circle-30.png\"))
   => true"
  {:added "2.8"}
  [obj]
  (extends? image/IRepresentation (type obj)))

(defn image-channels
  "returns the channels of an image
 
   (-> (image/read \"resources/data/circle-30.png\")
       (image-channels))
   ;;[#object[\"[B\" 0x76926cfc \"[B@76926cfc\"]]
   "
  {:added "2.8"}
  [image]
  (image/-channels image))

(defn image-size
  "returns the size of the image
   
   (-> (image/read \"resources/data/circle-30.png\")
       (image-size))
   => {:height 30, :width 30}"
  {:added "2.8"}
  [image]
  (image/-size image))

(defn image-model
  "returns the model associated with the image
 
   (-> (image/read \"resources/data/circle-30.png\")
       (image-model))
   => (model/model :4-byte-abgr)"
  {:added "2.8"}
  [image]
  (image/-model image))

(defn image-data
  "returns the raw data associated with the image
 
   (->> (image/read \"resources/data/circle-30.png\")
        (image-data)
        (take 10))
   => [-1 -1 -1 -1 -1 -1 -1 -1 -1 -1]"
  {:added "2.8"}
  [image]
  (image/-data image))

(defn size?
  "checks whether object is a size
 
   (size? [10 10]) => true
 
   (size? {:height 10 :width 10}) => true"
  {:added "2.8"}
  [obj]
  (cond (map? obj)
        (-> (keys obj) set (= #{:height :width}))

        (vector? obj)
        (-> (count obj) (= 2))

        :else
        (extends? image/ISize (type obj))))

(defn height
  "returns the height of an image
 
   (-> (image/read \"resources/data/circle-30.png\")
       (image/height))
   => 30"
  {:added "2.8"}
  [obj]
  (cond (image? obj)
        (image/-height (image-size obj))
        
        :else
        (image/-height obj)))

(defn width
  "returns the width of an image
 
   (-> (image/read \"resources/data/circle-30.png\")
       (image/width))
   => 30"
  {:added "2.8"}
  [obj]
  (cond (image? obj)
        (image/-width (image-size obj))
        
        :else
        (image/-width obj)))

(defn subimage
  "returns a subimage given image and bounds
   (-> (image/read \"resources/data/circle-30.png\")
       (subimage 10 10 10 10)
       (image-size))
   ;; #image.awt[10 10]{:model :4-byte-abgr}
   => {:height 10, :width 10}"
  {:added "2.8"}
  [image x y w h]
  (image/-subimage image x y w h))

(defn blank
  "returns a blank image
 
   (image/blank [10 10])
   ;;#image.awt[10 10]{:model :int-argb}
   => java.awt.image.BufferedImage"
  {:added "2.8"}
  ([size]
   (blank size *default-model* *default-type*))
  ([size model]
   (blank size model *default-type*))
  ([size model type]
   (image/-blank size model type)))

(defn read
  "reads an image from file
 
   (image/read \"resources/data/circle-30.png\"
               (model/model :byte-gray)
               :base)
   ;;#image.base[30 30]{:model :byte-gray}
   => hara.image.base.Image"
  {:added "2.8"}
  ([source]
   (read source *default-model* *default-type*))
  ([source model]
   (read source model *default-type*))
  ([source model type]
   (image/-read source model type)))

(defn to-byte-gray
  "converts an image :byte-gray representation
 
   (-> (image/read \"resources/data/circle-30.png\")
       (to-byte-gray))
   ;;#object[\"[B\" 0x2c1a4f3 \"[B@2c1a4f3\"]
   "
  {:added "2.8"}
  [image]
  (image/-to-byte-gray image))

(defn to-int-argb
  "converts an image :int-argb representation
 
   (-> (image/read \"resources/data/circle-30.png\")
       (to-int-argb))
   ;;#object[\"[I\" 0x3514a0dc \"[I@3514a0dc\"]
   "
  {:added "2.8"}
  [image]
  (image/-to-int-argb image))

(defn write
  "writes an image to file
 
   (-> (image/read \"resources/data/circle-30.png\")
      (image/coerce (model/model :3-byte-bgr) :base)
       (image/write \"resources/data/circle-30.jpg\" {:format \"JPG\"}))"
  {:added "2.8"}
  ([image sink]
   (write image sink {:format "png"}))
  ([image sink opts]
   (image/-write image sink opts)))

(defn image
  "creates an image given parameters
 
   (image {:size [2 2]
           :model :byte-gray
           :data (byte-array [1 2 3 4])})
   ;;#image.awt[2 2]{:model :byte-gray}
   => *default-type*"
  {:added "2.8"}
  ([{:keys [size model data type]}]
   (image/-image size
                 (or model *default-model*)
                 data
                 (or type *default-type*)))
  ([size model data]
   (image/-image size model data *default-type*))
  ([size model data type]
   (image/-image size model data type)))

(defn coerce
  "coerces one image to another
 
   (-> (image/read \"resources/data/circle-30.png\")
       (image/coerce :base))
   ;; #image.base[30 30]{:model :4-byte-abgr}
   => hara.image.base.Image"
  {:added "2.8"}
  ([image type]
   (let [size  (image-size image)
         model (image-model image)
         data  (image-data image)]
     (image/-image size model data type)))
  ([image to-model type]
   (let [base (common/convert image to-model)]
     (coerce base type))))

(defn display-class
  "shows which types can be displayed
 
   (display-class :base)
   => #{hara.image.base.Image}
   
   (display-class :awt)
   => #{java.awt.image.BufferedImage}"
  {:added "2.8"}
  ([type]
   (image/-display-class type)))

(defn display
  "displays an image
 
   (-> (image/read \"resources/data/circle-10.png\")
      (display {:channel :red}))"
  {:added "2.8"}
  ([image]
   (display image {}))
  ([image opts]
   (display image opts *default-view*))
  ([image opts type]
   (let [types (display-class type)
         image (if (some #(instance? % image) types)
                 image
                 (coerce image type))]
     (image/-display image opts type))))

(defn default-type
  "displays and sets the default type
   
   (default-type hara.image.base.Image)
 
   (default-type java.awt.image.BufferedImage)
 
   (default-type)
   => java.awt.image.BufferedImage"
  {:added "2.8"}
  ([] *default-type*)
  ([v]
   (alter-var-root #'*default-type*
                   (fn [_] v))))

(defn default-model
  "displays and set the default model
 
   (:label (default-model))
   => :int-argb"
  {:added "2.8"}
  ([] *default-model*)
  ([v]
   (alter-var-root #'*default-model*
                   (fn [_] v))))

(defn default-view
  "displays and set the default view
 
   (default-view)
   => :base"
  {:added "2.8"}
  ([] *default-view*)
  ([v]
   (alter-var-root #'*default-view*
                   (fn [_] v))))
