(ns hara.event.condition.data
  (:require [hara.event.common :as common]
            [hara.common.primitives :refer [uuid]]))

(defn issue
  "creates a new issue
 
   (issue {:a 1} \"hello\" {} nil)
   => (contains {:id keyword?
                 :data {:a 1},
                 :msg \"hello\",
                 :options {},
                 :optmap {},
                :default nil})"
  {:added "2.1"}
  ([data msg options default]
   (issue (common/new-id) data msg options default))
  ([id data msg options default]
   (let [data    (common/expand-data data)
         options (or options {})
         optmap  (zipmap (keys options) (repeat id))]
     {:id id
      :data data
      :msg msg
      :options options
      :optmap optmap
      :default default})))

(defn catch-condition
  "creates a catch condition
 
   (catch-condition :<TARGET> \"some value\")"
  {:added "2.1"}
  [target value]
  (ex-info "catch" {::condition :catch :target target :value value}))

(defn choose-condition
  "creates a choose-condition
   
   (choose-condition :<TARGET> :choice-A [1])"
  {:added "2.1"}
  [target label args]
  (ex-info "choose" {::condition :choose :target target :label label :args args}))

(defn exception
  "creates an exception
 
   (exception (issue {:a 1} \"hello\" {} nil))"
  {:added "2.1"}
  ([issue]
   (let [contents (:data issue)
         msg    (str (:msg issue) " - " contents)
         error  ^Throwable (ex-info msg contents)]
     (doto error
       (.setStackTrace (->> (seq (.getStackTrace error))
                            (filter (fn [^StackTraceElement name] (not (.startsWith (.getClassName name) "hara.event"))))
                            (into-array StackTraceElement))))))
  ([issue data]
   (exception (update-in issue [:data] merge data))))
