﻿using System;
using System.Globalization;
using System.Text.RegularExpressions;
using IUA.Business.Repository;

namespace IUA.Business.Validators
{
    public interface IBankAccountValidator
    {
        int Validate(string branchCode, string accountType, string accountNumber);
    }

    public class BankAccountValidator : IBankAccountValidator
    {
        /*    
         * FYI
         * public enum BankCode
            {
                StandardBankOfSouthAfricaLtd = 1,
                NedbankLimited = 2,
                FirstrandBank = 3,
                PeopleBankLtdIncPepBank = 5,
                BankOfAthensSaLtd = 6,
                NedbankLtdIncBoeBank = 8,
                MercantileBankLimited = 9,
                CapitecBankLimited=10,
                MtnBankingStandardBank = 11,
                SaReserveBank = 12,
                SouthAfricanPostOffice = 13,
                BankWindhoekBeperk = 14,
                AbsaForSamosUseOnly = 15,
                AbsaBank = 16,
                StandardBankSwazilandLtd = 17,
                TebaBankLimited = 19,
                ExSwaStillOnVisa = 21,
                HabibOverseasBankLimited = 22,
                LesothoBankLimited = 23,
                PeoplesBankLtdIncNbs = 24,
                PermanentBank = 26,
                FbcFidelityBankLtd = 27,
                UnibankLimited = 28,
                AbnAmroBank = 29,
                AbsaIthala = 30,
                ExSdsbSwazi = 31,
                ExCommunity = 32,
                Citibank = 33,
                BarclaysBankPlcSaBranch = 34,
                HbzBankLimited = 36,
                NedbankSwazilandLimited = 37,
                NedbankLesothoLimited = 38,
                InvestecBankLimited = 39,
                StandardCharteredBankSa = 42,
                NedbankNamibia = 43,
                BidvestBankLimited = 44,
                SocieteGeneralJhbBranch = 45,
                StandardBankLesotho = 46,
                GrindrodBankLimited = 47
            }*/

        /*  Returns an integer based on the validations
         *  -1: Successful (This also occurs when a bank cannot be validated)
         *
         */
        public int Validate(string branchCode, string accountType, string accountNumber)
        {
            if (IsAlphaNumeric(accountNumber))
            {
                return 2;
            }

            var bankValidationRepository = new BankValidationRepository();
            var bankDetails = bankValidationRepository.GetBankBranch(branchCode);

            if (bankDetails?.CDVBank == null) { return 1; }
            var bankBranchCode = bankDetails.BranchCode;
            var bankCode = bankDetails.CDVBank.BankCode;
            var validateInd = bankDetails.CDVBank.ValidateInd;
            var weighting = bankDetails.CDVBank.Weighting;
            var fudgeFactor = bankDetails.CDVBank.FudgeFactor.GetValueOrDefault();
            var modulus = bankDetails.CDVBank.Modulus.GetValueOrDefault();
            var exceptionCode = bankDetails.CDVBank.ExceptionCode;


            //Account type must be in the following types
            switch (accountType.ToUpper())
            {
                case "CHEQUE":
                    break;
                case "CURRENT":
                    break;
                case "SAVINGS":
                    break;
                case "TRANSMISSION":
                    break;
                default:
                    return 3;
            }

            //If the account numbers length is less than 7 or greater than 11, return a failure
            if (accountNumber.Length < 7 || accountNumber.Length > 11)
            {
                return 2;
            }

            //If the validateInd is set to 0, there is no validation for this bank, return a success
            if (validateInd == 0)
            {
                return -1;
            }

            var leastSignificantChar = accountNumber.Substring(accountNumber.Length - 1, 1);
            var secondLeastSignificantChar = accountNumber.Substring(accountNumber.Length - 2, 1);
            var mostSignificantChar = accountNumber.Substring(0, 1);
            var leastSignificantDigit = Convert.ToInt32(leastSignificantChar);


            var checkResult = 0;
            /*Bank Specific rules
             *Each bank has its own set of validation
             */
            switch (bankCode)
            {
                //Standard Bank

                case "0001":

                    checkResult = CheckAccountNumberStandardBank(accountNumber.Length, mostSignificantChar, bankBranchCode);
                    break;

                //ABN Amro Bank, African Bank, Investec Bank, Grindrod Bank, Post Bank, Reserve Bank,MTN Banking, Standard Charted Bank, Teba Bank
                case "0029":
                case "0007":
                case "0039":
                case "0047":
                case "0013":
                case "0012":
                case "0011":
                case "0042":
                case "0019":

                    checkResult = CheckAccountNumberInvestecBankGroup(accountNumber.Length, mostSignificantChar, bankCode);
                    break;

                //Absa Bank
                case "0016":
                    checkResult = CheckAccountNumberAbsaBank(accountNumber.Length);
                    break;

                //Capitec Bank, Mercantile Bank
                case "0010":
                case "0009":
                    checkResult = CheckAccountNumberMercantileBankGroup(accountNumber.Length);
                    break;

                //First Rand Bank, Habib Overseas Bank, NedBank, NedBank (BOE), UniBank
                case "0003":
                case "0022":
                case "0002":
                case "0008":
                case "0028":
                    checkResult = CheckAccountNumberNedbankGroup(accountNumber.Length);
                    break;

                //HBZ Bank
                case "0036":
                    checkResult = CheckAccountNumberHbzBank(accountNumber.Length);
                    break;

                //SA Bank of Athens
                case "0006":
                    checkResult = CheckAccountNumberSaBankOfAthens(accountNumber.Length);
                    break;

            }
            if (checkResult != -1)
            {
                return checkResult;
            }

            //Work out the Check Digit Verification (CDV)
            if (exceptionCode == null)
            {
                //If the Remainder is zero, then the account number is valid, otherwise it's not
                var remainder = CalculateCdvRemainder(accountNumber, weighting, fudgeFactor, modulus);
                if (remainder == 0) { return -1; }
                return 2;
            }

            //Work through each of the banks one at a time checking their exceptions

            switch (bankCode)
            {
                //Standard Bank
                case "0001":
                    checkResult = CheckDigitVerificationStandardBank(exceptionCode, accountNumber, bankBranchCode, mostSignificantChar, weighting, fudgeFactor, modulus);
                    break;

                //Nedbank Limited and Nedbank (BOE)
                case "0002":
                case "0008":
                    checkResult = CheckDigitVerificationNedbank(exceptionCode, accountType, accountNumber, weighting, modulus);
                    break;

                //ABSA
                case "0016":
                    checkResult = CheckDigitVerificationAbsa(exceptionCode, accountNumber, accountType, leastSignificantChar, leastSignificantDigit);
                    break;


                //Mercantile Bank
                case "0009":
                    checkResult = CheckDigitVerificationMercantileBank(exceptionCode, accountNumber, leastSignificantChar, weighting, fudgeFactor, modulus);
                    break;

                //Teba Bank
                case "0019":
                    checkResult = CheckDigitVerificationTebaBank(exceptionCode, bankBranchCode, accountNumber, fudgeFactor, modulus);
                    break;

                //Grindrod Bank Limited
                case "0047":
                    checkResult = CheckDigitVerificationGrindrodBank(exceptionCode, accountNumber, leastSignificantChar, secondLeastSignificantChar, weighting, fudgeFactor, modulus);
                    break;

                //HBZ Bank Limited
                case "0036":
                    checkResult = CheckDigitVerificationHbzBank(exceptionCode, accountNumber, leastSignificantChar, mostSignificantChar, weighting, fudgeFactor, modulus);
                    break;

                //Habib Bank
                case "0022":
                    checkResult = CheckDigitVerificationHabibBank(exceptionCode, accountNumber, accountType, leastSignificantDigit, mostSignificantChar, weighting, fudgeFactor, modulus);
                    break;
            }
            return checkResult;

        }

        private int CheckDigitVerificationHabibBank(string exceptionCode, string accountNumber, string accountType, int leastSignificantDigit, string mostSignificantChar, string weighting, int fudgeFactor, int modulus)
        {
            if (exceptionCode == "j")
            {
                var remainder = CalculateCdvRemainder(accountNumber, weighting, fudgeFactor, modulus);
                if (remainder == 0) { return -1; }

                if (accountNumber.Length == 11 && leastSignificantDigit > 0 &&
                    (accountType.ToUpper() == "CHEQUE" || accountType.ToUpper() == "CURRENT" ||
                     accountType.ToUpper() == "SAVINGS"))
                {
                    if (remainder == 1) { return -1; }
                }

                if (accountNumber.Length == 10 && mostSignificantChar == "0") { return -1; }

                return 2;
            }

            //ExceptionCode must be 'j', if not return an invalid
            return 0;
        }

        private int CheckDigitVerificationHbzBank(string exceptionCode, string accountNumber, string leastSignificantChar, string mostSignificantChar, string weighting, int fudgeFactor, int modulus)
        {
            if (exceptionCode == "d")
            {
                var remainder = CalculateCdvRemainder(accountNumber, weighting, fudgeFactor, modulus);
                if (remainder == 0) { return -1; }

                //NOTE: For New 11 digit Current and Savings Accounts, The remainder may be 1 if the least significant digit of the account number is 0 or 1
                if (accountNumber.Length == 11 && (leastSignificantChar == "0" || leastSignificantChar == "1"))
                {
                    if (remainder == 1)
                    {
                        return -1;
                    }
                }

                if ((accountNumber.Length == 8 || accountNumber.Length == 10) && mostSignificantChar == "0")
                {
                    return -1;
                }

                return 2;
            }
            //ExceptionCode must be 'd', if not return an invalid
            return 0;
        }

        private int CheckDigitVerificationGrindrodBank(string exceptionCode, string accountNumber, string leastSignificantChar, string secondLeastSignificantChar, string weighting, int fudgeFactor, int modulus)
        {
            if (exceptionCode == "k")
            {

                var remainder = CalculateCdvRemainder(accountNumber, weighting, fudgeFactor, modulus);
                if (remainder == 0) { return -1; }

                if (leastSignificantChar == secondLeastSignificantChar) { return -1; }
                return 2;
            }
            //ExceptionCode must be 'k', if not return an invalid
            return 0;

        }

        private int CheckDigitVerificationTebaBank(string exceptionCode, string bankBranchCode, string accountNumber, int fudgeFactor, int modulus)
        {
            if (exceptionCode == "2")
            {
                var remainder = 999; //To ensure that if the bankBranch is not within parameters, the Validation fails
                var bankBranchCodeInteger = Convert.ToInt32(bankBranchCode);

                if (bankBranchCodeInteger >= 431000 && bankBranchCodeInteger <= 431979)
                {
                    remainder = CalculateCdvRemainder(accountNumber, "19876543211", fudgeFactor, modulus);
                }

                if (bankBranchCodeInteger >= 431980 && bankBranchCodeInteger <= 431999)
                {
                    remainder = CalculateCdvRemainder(accountNumber, "27654321000", fudgeFactor, modulus);
                }

                if (remainder == 0) { return -1; }
                return 2;

            }
            //ExceptionCode must be '2', if not return an invalid
            return 0;
        }

        private int CheckDigitVerificationMercantileBank(string exceptionCode, string accountNumber, string leastSignificantChar, string weighting, int fudgeFactor, int modulus)
        {
            if (exceptionCode == "b")
            {
                var remainder = CalculateCdvRemainder(accountNumber, weighting, fudgeFactor, modulus);
                if (remainder == 0) { return -1; }
                //The remainder may be 1 if the least significant digit of the account number is 0 or 1. 
                if (leastSignificantChar == "0" || leastSignificantChar == "1")
                {
                    if (remainder == 1) { return -1; }
                    return 2;
                }
            }
            //ExceptionCode must be 'b', if not return an invalid
            return 0;
        }

        private int CheckDigitVerificationAbsa(string exceptionCode, string accountNumber, string accountType, string leastSignificantChar, int leastSignificantDigit)
        {
            if (exceptionCode == "f")
            {
                int remainder;
                if (accountNumber.Length == 11 && accountNumber.Substring(0, 2) == "53" &&
                    accountType.ToUpper() == "SAVINGS")
                {
                    remainder = CalculateCdvRemainder(accountNumber, "00000000000", 0, 0);
                    if (remainder == 0) { return -1; }
                    return 2;
                }

                remainder = CalculateCdvRemainder(accountNumber, "17329874321", 0, 10);
                if (remainder == 0) { return -1; }

                remainder = CalculateCdvRemainder(accountNumber, "14327654321", 0, 11);
                if (remainder == 0) { return -1; }

                remainder = CalculateCdvRemainder(accountNumber, "54327654321", 0, 11);
                if (remainder == 0) { return -1; }

                if ((leastSignificantChar == "0" || leastSignificantChar == "1") && (accountNumber.Length == 10 || accountNumber.Length == 11))
                {
                    if (remainder == 1) { return -1; }
                }

                remainder = CalculateCdvRemainder(accountNumber, "11327654321", 0, 11);
                if (remainder == 0) { return -1; }

                if (accountNumber.Length < 10)
                {
                    leastSignificantDigit += 6;
                    leastSignificantChar =
                    leastSignificantDigit.ToString(CultureInfo.InvariantCulture)
                                             .Substring(leastSignificantDigit.ToString(CultureInfo.InvariantCulture).Length - 1, 1);

                    var modifiedAccountNumber = accountNumber.Substring(0, accountNumber.Length - 1) +
                                            leastSignificantChar;

                    remainder = CalculateCdvRemainder(modifiedAccountNumber, "11327654321", 0, 11);
                    if (remainder == 0) { return -1; }
                }

                remainder = CalculateCdvRemainder(accountNumber, "14329874321", 0, 10);
                if (remainder == 0) { return -1; }
                return 2;
            }
            //Exception Code must be "f", If not, return failure
            return 0;
        }

        private int CheckDigitVerificationNedbank(string exceptionCode, string accountType, string accountNumber, string weighting, int modulus)
        {
            if (exceptionCode == "1")
            {
                int remainder;
                if (accountType.ToUpper() == "CHEQUE" || accountType.ToUpper() == "CURRENT")
                {
                    remainder = CalculateCdvRemainder(accountNumber, weighting, 9, modulus);
                }
                else
                {
                    remainder = CalculateCdvRemainder(accountNumber, weighting, 18, modulus);
                }

                if (remainder == 0) { return -1; }
                return 2;
            }
            //Exception Code must be "1", If not, return failure
            return 0;

        }

        private int CheckDigitVerificationStandardBank(string exceptionCode, string accountNumber, string bankBranchCode, string mostSignificantChar, string weighting, int fudgeFactor, int modulus)
        {
            if (exceptionCode == "m") //TODO: Possibly remove this logic as I do not see it in the Bankserv documentation
            {
                int remainder;
                if (accountNumber.Length == 11 && bankBranchCode == "051001" && mostSignificantChar == "1")
                {
                    remainder = CalculateCdvRemainder(accountNumber, "DC987654321", 0, 11);
                }
                else
                {
                    remainder = CalculateCdvRemainder(accountNumber, weighting, fudgeFactor, modulus);
                }

                if (remainder == 0) { return -1; }
                return 2;
            }
            //Exception Code must be "m", If not, return failure
            return 0;

        }

        private int CheckAccountNumberSaBankOfAthens(int length)
        {
            if (length != 7 && length != 11) { return 2; }
            return -1;
        }

        private int CheckAccountNumberHbzBank(int length)
        {
            if (length < 8 || length > 11 || length == 9) { return 2; }
            return -1;
        }

        private int CheckAccountNumberNedbankGroup(int length)
        {
            if (length < 10 || length > 11) { return 2; }

            return -1;
        }

        private int CheckAccountNumberMercantileBankGroup(int length)
        {
            if (length != 10) { return 2; }
            return -1;
        }

        private int CheckAccountNumberAbsaBank(int length)
        {
            if (length < 8 || length > 11) { return 2; }
            return -1;

        }

        private int CheckAccountNumberInvestecBankGroup(int length, string mostSignificantChar, string bankCode)
        {
            if (length != 11) { return 2; }
            if (bankCode == "0011" && mostSignificantChar != "0") { return 2; }

            return -1;
        }

        private int CheckAccountNumberStandardBank(int length, string mostSignificantChar, string bankBranchCode)
        {
            if (length != 9 && length != 11) { return 2; }
            if (length == 11 && mostSignificantChar != "1") { return 2; }
            if (length == 11 && bankBranchCode != "051001") { return 2; }

            return -1;
        }

        private int CalculateCdvRemainder(string accountNumber, string weighting, int fudgeFactor, int modulus)
        {
            var remainder = 0;
            var accountNumberDifference = 11 - accountNumber.Length;
            var weightingDifference = 11 - weighting.Length;
            var sumOfProducts = 0;


            if (accountNumberDifference > 0)
            {
                accountNumber = "00000000000" + accountNumber;
                accountNumber = accountNumber.Substring(11 - accountNumberDifference, 11);
            }

            if (weightingDifference > 0)
            {
                weighting = "00000000000" + weighting;
                weighting = weighting.Substring(11 - weightingDifference, 11);
            }

            var charCount = 0;

            while (charCount <= 10)
            {
                var accountChar = accountNumber.Substring(charCount, 1);
                var weightingChar = weighting.Substring(charCount, 1);
                var product = MultiplyWeightingAndAccountCharacter(accountChar, weightingChar);
                sumOfProducts += product;
                charCount++;
            }

            if (modulus != 0) { remainder = (sumOfProducts + fudgeFactor) % modulus; }
            return remainder;

        }

        private int MultiplyWeightingAndAccountCharacter(string accountNumberChar, string weightingChar)
        {
            int weightingValue;
            var isNumeric = int.TryParse(weightingChar, out weightingValue);

            if (!isNumeric)
            {
                switch (weightingChar.ToUpper())
                {
                    case "A":
                        weightingValue = 10;
                        break;
                    case "D":
                        weightingValue = 13;
                        break;
                    case "H":
                        weightingValue = 17;
                        break;
                    case "J":
                        weightingValue = 19;
                        break;
                    case "N":
                        weightingValue = 23;
                        break;
                    case "T":
                        weightingValue = 29;
                        break;
                }
            }
            var accountValue = Convert.ToInt32(accountNumberChar);
            var product = accountValue * weightingValue;

            return product;


        }

        private bool IsAlphaNumeric(string strToCheck)
        {
            var rg = new Regex(@"[a-zA-Z\s]");
            return rg.IsMatch(strToCheck);
        }
    }
}
