(ns simply.banking-details.core
  (:require [simply.errors :as e]
            [simply.banking-details.banks :as banks]
            #?(:clj  [clojure.spec.alpha :as s]
               :cljs [cljs.spec.alpha :as s])
            [clojure.string :as string]))


(defn valid-branch-code? [c]
  (boolean
   (and (string? c)
        (= (count c) 6)
        (re-find #"^[0-9]*$" c))))


(defn validate-branch-code [c]
  (if (valid-branch-code? c)
    (e/validation)
    (e/validation "A Branch Code should be 6 numbers")))


(defn valid-account-number? [c]
  (boolean
   (and (string? c)
        (> (count c) 6)
        (> 12 (count c))
        (re-find #"^[0-9]*$" c))))


(defn validate-account-number [c]
  (if (valid-account-number? c)
    (e/validation)
    (e/validation "Only numbers allowed and should have a length between 7 and 11")))


(defn valid-debit-day? [d]
  (and
   (int? d)
   (> d 0)
   (< d 32)))


(defn validate-debit-day [d]
  (if (valid-debit-day? d)
    (e/validation)
    (e/validation "Only numbers 1 to 31 are valid debit days.")))


(def bank-account-types #{"Cheque" "Savings" "Transmission" "Current"})


(defn valid-bank-account-type? [t]
  (contains? bank-account-types t))

(defn validate-bank-account-type [t]
  (if (valid-bank-account-type? t)
    (e/validation)
    (e/validation "Only account types 'Cheque', 'Savings', 'Current', and 'Transmission' are allowed")))


(defmulti account-number-matches-branch-code? (fn [bank-account-group bank account-number] bank-account-group))


(defn validate-banking-details [branch-code account-number]
  (let [valid (e/validation)
        invalid e/validation
        invalid-account-number (invalid "Account Number and Branch Code do not match")
        bank (banks/bank-from-branch-code branch-code)
        bank-account-group (get banks/bank-code-bank-account-groups-map (:bank-code bank))]
    (cond
      (not (string? account-number))
      (invalid "Account Number should be a string")

      (not (re-find #"^[0-9]*$" account-number))
      (invalid "Account Number can only contain numbers")

      (< (count account-number) 7)
      (invalid "Account Number should have more than 6 numbers")

      (> (count account-number) 11)
      (invalid "Account Number should have less than 12 numbers")

      (nil? bank)
      (invalid "Unknown Branch Code")

      (= 0 (:validate-ind bank))
      valid

      (nil? bank-account-group)
      invalid-account-number

      (not (account-number-matches-branch-code? bank-account-group bank account-number))
      invalid-account-number

      :else
      valid)))


(defn- lenght-of? [l s] (= (count s) l))


(defmethod account-number-matches-branch-code? :standard-bank
  [_ bank account-number]
  (or (lenght-of? 9 account-number)
      (and (lenght-of? 11 account-number)
           (string/starts-with? account-number "1")
           (= (:branch-code bank) "051001"))))


(defmethod account-number-matches-branch-code? :sa-bank-of-athens
  [_ _ account-number]
  (or (lenght-of? 7 account-number)
      (lenght-of? 11 account-number)))


(defmethod account-number-matches-branch-code? :hbz-group
  [_ _ account-number]
  (or (lenght-of? 8 account-number)
      (lenght-of? 10 account-number)
      (lenght-of? 11 account-number)))


(defmethod account-number-matches-branch-code? :nedbank-group
  [_ _ account-number]
  (or (lenght-of? 10 account-number)
      (lenght-of? 11 account-number)))


(defmethod account-number-matches-branch-code? :mercantile-bank-group
  [_ _ account-number]
  (lenght-of? 10 account-number))


(defmethod account-number-matches-branch-code? :absa
  [_ _ account-number]
  (or (lenght-of? 8 account-number)
      (lenght-of? 9 account-number)
      (lenght-of? 10 account-number)
      (lenght-of? 11 account-number)))


(defmethod account-number-matches-branch-code? :investec-bank-group
  [_ bank account-number]
  (if (= (:bank-code bank) "11")
    (and (lenght-of? 11 account-number)
         (string/starts-with? account-number "0"))
    (lenght-of? 11 account-number)))


(defmethod account-number-matches-branch-code? :tyme-bank
  [_ bank account-number]
  (lenght-of? 11 account-number))


(defmethod account-number-matches-branch-code? :discovery-bank
  [_ bank account-number]
  (lenght-of? 11 account-number))


(defmethod account-number-matches-branch-code? :default [_ _ _] false)
