(ns simply.threads)

(def ^:private *thread-properties (atom {}))

(defn- current-thread-id []
  (.getId (Thread/currentThread)))


(defn set-property! [k v]
  (let [id (current-thread-id)]
    (swap! *thread-properties assoc-in [id k] v)
    {:thread-id id
     :key k
     :value v}))


(defn get-property [k]
  (let [id (current-thread-id)
        v (get-in @*thread-properties [id k])]
    (when v
      {:thread-id id
       :key k
       :value (get-in @*thread-properties [id k])})))


(defn clear-property! [prop]
  (swap! *thread-properties
         (fn [props]
           (let [{:keys [thread-id key]} prop
                 thread (-> (get props thread-id)
                            (dissoc key))]
             (if (empty? thread)
               (dissoc props thread-id)
               (assoc props thread-id thread)))))
  nil)


(comment
  "USAGE"
  (def prop (set-property! :e 2))
  (get-property :e)
  (clear-property! (get-property :e))

  )


(comment

  (defn number-of-threads [r]
    (->> r
         (map #(drop 1 %))
         (reduce into)
         (map :thread-id)
         set
         count))

  (defn values-as-expected [r]
    (->> r
         (map (fn [[a b c]]
                (and (= a (:value b) (:value c))
                     (= (:thread-id b) (:thread-id c)))))
         (every? true?)))


  "PMAP TEST:
   iterate over a 1000 items,
   each with random sleep duration
   and confirm the correct values are returned by set stat"
  (def pmap-test-result
    (->> (range 1000)
         (pmap (fn [i]
                 (let [s (set-property! :i i)]
                   (Thread/sleep (rand-int 3000))
                   [i s (get-property :i)])))
         (doall)))

  (number-of-threads pmap-test-result) ; => 35
  (values-as-expected pmap-test-result) ; => true


  "AT-AT TEST:
   run 100 at-at jobs that repeat every 10ms for a total duration of 20 seconds
   each run randomly sleeps for 0-99 ms, simulating our pubsub pulls"
  (require 'overtone.at-at)
  (def pool (overtone.at-at/mk-pool))
  (def *run-count (atom 0))
  (def *at-at-result (atom []))
  (defn job []
    (overtone.at-at/every
     10
     (fn []
       (let [i (swap! *run-count inc)]
         (let [s (set-property! :i i)]
           (Thread/sleep (rand-int 100))
           (swap! *at-at-result conj [i s (get-property :i)]))))
     pool))

  (do
    (reset! *run-count 0)
    (reset! *at-at-result [])
    (let [jobs (doall (repeatedly 100 job))]
      (Thread/sleep 20000)
      (doseq [j jobs]
        (overtone.at-at/stop j))))

  @*run-count ; 3786
  (number-of-threads @*at-at-result) ; 10
  (values-as-expected @*at-at-result) ; true

  )
