(ns simply.gcp.errorreporting
  (:require [cheshire.core :as json]
            [simply.http.client :as http]
            [simply.gcp.util :as util]
            [integrant.core :as ig]))


(defn- event-data [{:keys [?err msg_ instant ?ns-str ?line]} context]
  (let [msg_ @msg_
        msg (if-not (instance? Throwable ?err)
              (str msg_ " | " ?ns-str " | " ?line)
              (let [e (if (empty? msg_)
                        ?err
                        (ex-info msg_ {:ns ?ns-str :line ?line :e (type ?err)} ?err))
                    stack-writer (java.io.StringWriter.)]
                (.printStackTrace e (java.io.PrintWriter. stack-writer))
                (str stack-writer)))]
    {:message msg
     :eventTime (util/unparse-zulu-date-format instant)
     :serviceContext context
     :context {:reportLocation {:functionName (str ?ns-str)
                                :lineNumber (or ?line 0)}}}))


(defn- appender-fn
  [{:keys [project-id api-key]} context]
  (let [url (format
             "https://clouderrorreporting.googleapis.com/v1beta1/projects/%s/events:report?key=%s"
             project-id api-key)]
    (fn [timbre-data]
      (try
        (let [event (event-data timbre-data context)
              request
              {:method :post
               :url url
               :body (json/generate-string event)}]
          (http/request request))
        (catch Exception e)))))


(defn- get-context [{:keys [container-name container-id]}]
  {:service container-name
   :version container-id
   :resourceType "container"})


(def ^:private defaults {:enabled?   true
                         :async?     true
                         :min-level  :error
                         :rate-limit nil
                         :output-fn  :inherit})

(defn appender
  "timbre appender for GCP Error Reporting"
  [options]
  (assoc defaults :fn (appender-fn options (get-context options))))


(defmethod ig/init-key :simply.gcp.errorreporting/appender
  [_ {:keys [config] :as options}]
  (let [{:keys [project-id api-key container-name container-id]} config]
    {:name :clouderrors
     :appender (appender config)}))



(comment
  (require 'taoensso.timbre)


  "RUN TO INCLUDE A LOCAL TEST APPENDER THAT PRINTS"

  (defn f [timbre-data]
    (clojure.pprint/pprint (event-data timbre-data {})))

  (taoensso.timbre/merge-config!
   {:appenders {:test-errors
                (assoc defaults
                       :fn f
                       :async? false)}})


  "RUN TO INCLUDE A PRE-PROD TEST APPENDER"
  "REMEMBER TO ADD GCP_API_KEY (preferably use the one from pre-prod)"
  "LOOK FOR ERRORS ON GOOGLE CLOUD ERROR REPORTING"

  (taoensso.timbre/merge-config!
   {:appenders {:test-full
                (assoc defaults
                       :fn (appender-fn {:project-id "simply-pre-prod"
                                         :api-key ""}
                                        {:service "testing"
                                         :version "testing-1"
                                         :resourceType "container"})
                       :async? false)}})

  "RUN TO TEST YOUR APPENDERS"
  (taoensso.timbre/error (Exception. "B1"))
  (taoensso.timbre/error (ArithmeticException. "E3") "AE2")
  (taoensso.timbre/error (ex-info "B3" {}) "A3")
  (taoensso.timbre/error (ex-info "B4" {}))
  (taoensso.timbre/error "A5 with a poop")


  taoensso.timbre/*config*

  )
