(ns simply.gcp.datastore.indexes
  "Datastore indexes as code.
   Will check all existing indexes then:
     - notify if any are in ERROR status
     - create non existing ones.

  To add to your project, add the following to your integrant system.edn config

  {:simply.gcp.datastore.indexes/create
   {:client #ig/ref :your-client-provider
    :project-id #dyn/prop [GCP_PROJECT_ID \"dogmatix\"]
    :indexes [{:kind     \"contract-updates\",
               :ancestor \"NONE\",
               :properties
               [{:name \"transaction-number\", :direction \"ASCENDING\"}
                {:name \"update-date\", :direction \"DESCENDING\"}]}]
    :should-create? true}}

   for index info see
     doc - https://cloud.google.com/datastore/docs/tools/indexconfig
     api - https://cloud.google.com/datastore/docs/reference/admin/rest/v1/projects.indexes#Index"
  (:require [simply.errors :as e]
            [cheshire.core :as json]
            [clojure.set]
            [taoensso.timbre :as logger]
            [clojure.pprint]
            [integrant.core :as ig]))


(defn- indexes-list-request []
  {:method :get
   :url "/indexes"
   :query-params {:pageSize 0}})


(defn index-create-request [index]
  {:method :post
   :url "/indexes"
   :body index})


(defn- wrap-api [project-id client]
  (fn [request]
    (client
      (cond-> request
        (contains? request :body) (update :body json/generate-string)
        true (update :url #(str "https://datastore.googleapis.com/v1/projects/" project-id %))
        true (assoc :scope "https://www.googleapis.com/auth/datastore")))))


(defn- wrap-error-handling [client]
  (fn [request]
    (let [{:keys [status body error opts] :as response} @(client request)]
      (when (or (nil? status) (<= 300 status) (> 200 status))
        (e/throw-app-error "Datastore Admin Api Request Failed" {:status status
                                                                 :body   body
                                                                 :error  error
                                                                 :url    (:url opts)
                                                                 :method (:method opts)}))
      response)))


(defn- wrap-response [client]
  (fn [req]
    (let [response (client req)]
      (cond-> response
        (:body response) (update :body json/parse-string keyword)))))


(defn- api-client [project-id client]
  (->> client
       (wrap-api project-id)
       (wrap-error-handling)
       (wrap-response)))


(defn- indexes-from-response [response]
  (->> response
       :body
       :indexes))


(defn- warn-about-error-indexes [indexes]
  (let [errors (filter #(= "ERROR" (:state %)) indexes)]
    (when (seq errors)
      (logger/error (ex-info "Datastore Indexes in error state: Manually fix data, remove and readd" {:indexes indexes})))))


(defn- indexes-to-add [expected existing]
  (let [existing (->> existing
                      (map #(dissoc % :projectId :indexId :state))
                      set)]
    (vec (clojure.set/difference (set expected) existing))))


(defn- create-indexes [{:keys [project-id client indexes]}]
  (try
    (logger/info (str "CREATING INDEXES: " (count indexes)))
    (let [expected    indexes
          client      (api-client project-id client)
          existing    (->> (client (indexes-list-request))
                           indexes-from-response)
          new-indexes (indexes-to-add expected existing)]
      (warn-about-error-indexes existing)
      (doseq [index new-indexes]
        (logger/info (str "CREATING INDEX: \n " (with-out-str (clojure.pprint/pprint index))))
        (try
          (client (index-create-request index))
          (catch Exception e
            (logger/error (ex-info "CREATING INDEX FAILED" {:index index} e))))))
    (catch Exception e
      (logger/error (ex-info "CREATING INDEXES FAILED" {:indexex indexes
                                                        :project-id project-id} e)))))


(defmethod ig/init-key :simply.gcp.datastore.indexes/create
  [_ {:keys [should-create?] :as options}]
  (when should-create?
    (create-indexes options)))


(comment
  (require '[simply.helpers])

  (def example-index
    {:kind     "contract-updates",
     :ancestor "NONE",
     :properties
     [{:name "transaction-number", :direction "ASCENDING"}
      {:name "update-date", :direction "DESCENDING"}]})

  (let [env :pre-prod
        client (api-client (simply.helpers/->project-id env) (simply.helpers/->client env))]
    (->> (client (indexes-list-request))
         indexes-from-response)
    )

  ,)
