(ns simply.spec.whatsapp
  (:require [clojure.spec.alpha :as s]))

(defn phone-number? [n]
  (boolean (re-matches #"^(27)[0-9]{9}$" n)))

(defn date-iso8601-utc? [d]
  (boolean
   (re-matches #"^[0-9]{4}-[0-9]{2}-[0-9]{2}T[0-9]{2}:[0-9]{2}:[0-9]{2}\.[0-9]{3,4}Z$" d)))

(s/def ::to phone-number?)
(s/def ::id string?)
(s/def ::content string?)
(s/def ::template string?)
(s/def ::created date-iso8601-utc?)
(s/def ::param string?)
(s/def ::params (s/coll-of ::param))
(s/def ::search-val string?)
(s/def ::reply keyword?)
(s/def ::search-map (s/keys :req-un [::search-val ::reply]))
(s/def ::reply-mapping (s/coll-of ::search-map))
(s/def ::replies (s/map-of keyword? ::message))
(s/def ::fallback (s/map-of keyword? map?))

(defmulti message-type :type)
(defmethod message-type :plain-message [_]
  (s/keys :req-un [::content]
          :opt-un [::reply-mapping]))

(defmethod message-type :template-message [_]
  (s/keys :req-un [::template ::params]
          :opt-un [::reply-mapping]))

(defmethod message-type :message-set [_]
  (s/keys :req-un [::id ::to ::created ::root]
          :opt-un [::replies ::fallback]))

(s/def ::message (s/multi-spec message-type :type))

(s/def ::root ::message)

(comment

  (s/explain-data ::message {:type :message-set
                             :to "27551234567"
                             :id "1"
                             :created "2020-08-25T12:17:29.499Z"
                             :root {:type :template-message
                                    :template "hi"
                                    :params ["oi"]}})

  (s/explain-data ::message {:id "simply-tenandsix-insured-not-owner-detail-request",
                             :type :message-set
                             :to "27551121122",
                             :root {:template "simply-tenandsix-insured-not-owner-detail-request",
                                    :type :template-message
                                    :reply-mapping [{:search-val "hi" :reply :something}]
                                    :params [" John Second" "fgfg" "Simply" "http://localhost:3467/p/JOHVAN200825FWHS" "47146"]},
                             :replies {:something {:type :plain-message
                                                   :content "hello"}}
                             :fallback {:sms {:type "employee-details",
                                              :number "0551121122",
                                              :text " John Second, fgfg has taken out an insurance policy for you from Simply. Go to http://localhost:3467/p/JOHVAN200825FWHS to complete application. (FSP:47146)",
                                               :reference "JOHVAN200825FWHS"}},
                             :created "2020-08-25T12:17:29.499Z"})

  )
