(ns simply.ops.tracing-db-reporter
  (:require [integrant.core :as ig]
            [manifold.bus :as m.bus]
            [manifold.stream :as m.stream]
            [simply.persistence.core :as p.db]
            [cheshire.core :as json]
            [hiccup.page :as hiccup]
            [taoensso.timbre :as logger]
            [simply.ops.tracing :as tracing]))


;; TO USE ADD to system.edn
;; :simply.ops.tracing-db-reporter/report-tracing-metrics-to-datastore
;; {:service "your-service"
;;  :report? #dyn/boolean #dyn/prop [SIMPLY_CORE_TRACING_TO_DATASTORE "true"]


(defn- report-tracing-metric [service i]
  (try
    (p.db/upsert
      (p.db/entity
        :db-namespace (p.db/db-namespace "OPS")
        :entity-key "tracing-metrics"
        :id (str (:time i) "-" (:id i))
        :data (assoc i :service service)))
    (catch Exception e
      (logger/error (ex-info "Cannot persist instance metric" i e)))))


(defn- start-reporting-tracing-metrics-to-datastore! [service]
  (->> (m.bus/subscribe tracing/tracing-report-bus tracing/tracing-report-topic)
       (m.stream/consume #(report-tracing-metric service %))))


(defmethod ig/init-key :simply.ops.tracing-db-reporter/report-tracing-metrics-to-datastore [_ {:keys [service report?]}]
  (when report?
    (start-reporting-tracing-metrics-to-datastore! service)))


;;;; DATA

(defn- get-request-metrics [request-id]
  (->> (p.db/find-all
         (p.db/query
           :db-namespace (p.db/db-namespace "OPS")
           :entity-key "tracing-metrics"
           :params {:request-id request-id}))
       (map #(-> % ::p.db/data (assoc :db-id (::p.db/id %))))))


(defn- get-request-events [request-id]
  (->> (p.db/find-all
         (p.db/query
           :db-namespace (p.db/db-namespace "Events")
           :entity-key "event"
           :params {:requestId request-id}))
       (map #(-> % ::p.db/data (dissoc :payload)
                 (assoc :trigger "EVENT"
                        :db-id (::p.db/id %)
                        :id (::p.db/id %))))))

;;;; REQUEST METRICS

(defn- recursively-apply-metric-children
  ([metrics-by-parents ms] (recursively-apply-metric-children 1 metrics-by-parents ms))
  ([n metrics-by-parents ms]
   (if (empty? ms)
     ms
     (->> ms
          (map (fn [{:keys [id] :as m}]
                 (let [children (get metrics-by-parents id [])
                       n' (inc n)]
                   (-> m
                       (assoc :level n
                              :children (recursively-apply-metric-children n' metrics-by-parents children))
                       (update :events (fn [e] (map #(assoc % :level n') e)))))))))))



(defn- unwind-trees [trees]
  (->> trees
       (map (fn [{:keys [events children] :as metric}]
              (->> [(dissoc metric :events :children)]
                   (into events)
                   (into (unwind-trees children)))))
       (reduce into)))


(defn- remove-required-actions [metrics]
  (let [metrics       (->> metrics
                           (map (fn [m]
                                  (assoc m :required-action?
                                         (and (clojure.string/starts-with? (:trigger m) "PUBSUB")
                                              (when-let [d (get-in m [:data :topic])]
                                                (some? (re-find #"required-action" d))))))))
        parent-lookup (into {} (map (juxt :id :parent-id) metrics))
        subs          (->> metrics
                           (filter #(and (= "PUBSUB_SUBSCRIPTION" (:trigger %))
                                         (:required-action? %)))
                           (map (fn [{:keys [id parent-id]}]
                                  [id (get parent-lookup parent-id)]))
                           (into {}))]
    (->> metrics
         (filter #(not (:required-action? %)))
         (map (fn [t]
                (if-let [parent (get subs (:parent-id t))]
                  (assoc t :parent-id parent)
                  t))))))


(defn- process-request-metrics [metrics events]
  (let [events-by-parent    (group-by :parent-id events)
        slowest-metric-time (->> metrics (sort-by :duration) last :duration)
        metrics             (sort-by :time metrics)
        orphaned-events     (get events-by-parent nil [])
        metrics-with-events (->> metrics
                                 remove-required-actions
                                 (map (fn [{:keys [id duration] :as m}]
                                        (assoc m
                                               :events (get events-by-parent id [])
                                               :duration-percentage (* 100 (double (/ duration slowest-metric-time)))))))
        metrics-by-parents  (group-by :parent-id metrics-with-events)
        top-level-metrics   (get metrics-by-parents nil [])
        trees               (recursively-apply-metric-children metrics-by-parents top-level-metrics)

        trigger-frequencies (->> (concat metrics events) (map :trigger) frequencies)
        full-metrics        (concat (unwind-trees trees) orphaned-events)]
    {:trigger-frequencies trigger-frequencies
     :metrics             full-metrics
     :metrics-by-duration (->> metrics-with-events (sort-by :duration-percentage) reverse)
     :entry-point         (first metrics)
     :top-level-metrics   top-level-metrics
     :orphaned-events     orphaned-events
     :trees               trees
     :duration            (- (:time (last metrics)) (:time (first metrics)))
     :depth               (:level (last (sort-by :level full-metrics)))}))


;;;; VISUALIZE REQUEST

(defn- color [d]
  (case (:trigger d)
    "COMMAND"             "#faa43a"
    "EVENT"               "#60bd68"
    "ACTION"              "#5da5da"
    "PUBSUB_SUBSCRIPTION" "#b276b2"
    "PUBSUB_TOPIC"        "#f15854"
    "#879da5"))


(defmulti ^:private node-opts :trigger)

(defmethod node-opts :default [d] {:label (:trigger d)})

(defmethod node-opts "COMMAND" [d]
  {:label (str "COMMAND\n" (clojure.string/lower-case (get-in d [:data :type])))})

(defmethod node-opts "EVENT" [d]
  {:label (str "EVENT\n\n" (clojure.string/lower-case (get-in d [:type])))})

(defmethod node-opts "ACTION" [d]
  {:label (str "ACTION\n\n" (clojure.string/lower-case (get-in d [:data :type])))})

(defmethod node-opts "PUBSUB_SUBSCRIPTION" [d]
  {:label (str "PULL\n\n"
               (when-let [t (get-in d [:data :type])]
                 (str (clojure.string/lower-case t) "\n\n"))
               (clojure.string/lower-case (get-in d [:data :sub])))})

(defmethod node-opts "PUBSUB_TOPIC" [d]
  {:label (str "PUSH(" (-> d :data :message-total) ")\n\n" (clojure.string/lower-case (get-in d [:data :topic])))})


(defn- processed-request-metrics-to-graph-network [processed-metrics]
  (let [{:keys [metrics]} processed-metrics
        nodes             (->> metrics
                               (map (fn [{:keys [id level service] :as e}]
                                      (cond-> (node-opts e)
                                          true (assoc :id id :level level :color (color e) :group service)
                                          true (update :label #(if service (str % "\n\n " service) %))))))
        edges             (->> metrics
                               (map (fn [{:keys [parent-id id] :as m}]
                                      (when parent-id
                                        {:from parent-id :to id :color (color m)})))
                               (remove nil?))]
    {:nodes nodes :edges edges}))


(defn- graph-component [{:keys [nodes edges] :as network}]
  (let [data-json    (json/generate-string network {:escape-non-ascii true})
        vis-opts     {:layout
                      {:hierarchical
                       {:enabled         true
                        :direction       "UD"
                        :sortMethod      "directed"
                        :shakeTowards    "leaves"
                        :edgeMinimization true
                        :levelSeparation 200}}
                      :edges {:arrows {:to {:enabled     true
                                            :type        "arrow"
                                            :scaleFactor 0.8}}
                              :smooth true}
                      :nodes {:shape           "box"
                              :widthConstraint {:maximum 67}
                              :margin          {:top 10 :bottom 10 :left 20 :right 20}}}
        options-json (json/generate-string vis-opts {:escape-non-ascii true})]
      [:div
       [:style "#network {height: 85vh}"]
       [:div#network]
       [:script {:type "text/javascript" :src "https://unpkg.com/vis-network/standalone/umd/vis-network.min.js"}]
       [:script {:type "text/javascript"}
        "var VIS_DATA = " data-json  ";"
        "var container = document.getElementById('network');"
        "var nodes = new vis.DataSet(VIS_DATA.nodes);"
        "var edges = new vis.DataSet(VIS_DATA.edges);"
        "var data = {nodes: nodes, edges: edges};"
        "var options = " options-json ";"
        "var network = new vis.Network(container, data, options);"]]))


(defn- trigger-frequency-view [processed-metrics]
  (let [{:keys [trigger-frequencies entry-point duration depth]} processed-metrics]
    (into
      [:div.field.is-grouped.is-grouped-multiline {:style "justify-content: center"}
       [:div.control
        [:div.tags.has-addons.are-small
         [:span.tag.is-info "trigger"]
         [:span.tag (:trigger entry-point)]]]
       [:div.control
        [:div.tags.has-addons.are-small
         [:span.tag.is-info "type"]
         [:span.tag (-> entry-point :data :type) ]]]
       [:div.control
        [:div.tags.has-addons.are-small
         [:span.tag.is-info "duration"]
         [:span.tag (format "%.3f sec" (double (/ duration 1000)))]]]
       [:div.control
        [:div.tags.has-addons.are-small
         [:span.tag.is-info "depth"]
         [:span.tag depth]]]]
      (->> trigger-frequencies
           (map (fn [[k v]]
                  [:div.control
                   [:div.tags.has-addons.are-small
                    [:span.tag.has-text-light {:style (str "background-color: " (color {:trigger k}))} (clojure.string/lower-case k)]
                    [:span.tag v]]]))))))


(defn- durations [processed-metrics]
  (let [{:keys [metrics-by-duration]} processed-metrics]
    (into
      [:div.is-size-7 {:style "width: 20vw"}]
      (->> metrics-by-duration
           (take 10)
           (map
             (fn [{:keys [trigger data duration-percentage duration]}]
               [:div
                [:strong (->> (clojure.string/split trigger #"_") (map first) (clojure.string/join "")) " "]
                [:span (:type data) " " (or (:sub data) (:topic data)) " "]
                [:strong.is-pulled-right (int (/ duration 1000000)) "ms"]
                [:progress.progress.is-small
                 {:value (Math/round duration-percentage) :max 100
                  :class (cond
                           (> duration-percentage 75) "is-danger"
                           (> duration-percentage 50) "is-warning"
                           :else "is-success")}]]))))))


(defn generate-request-viz
  ([request-id]
   (let [*metrics (future (get-request-metrics request-id))
         *events  (future (get-request-events request-id))]
     (generate-request-viz @*metrics @*events)))
  ([metrics events]
   (if (empty? metrics)
     (hiccup/html5 {} [:html [:body "No data for this request"]])
     (let [processed-metrics (process-request-metrics metrics events)
           network           (->> processed-metrics
                         processed-request-metrics-to-graph-network
                         graph-component)]
       (hiccup/html5
         {}
         [:html
          [:head
           [:meta {:charset "utf-8"}]
           [:meta {:name "viewport" :content "width=device-width, initial-scale=1"}]
           [:link {:rel  "stylesheet"
                   :href "https://cdn.jsdelivr.net/npm/bulma@0.9.2/css/bulma.min.css"}]]


          [:body
           [:section.section
            [:div.has-text-centered
             (trigger-frequency-view processed-metrics)]
            [:hr]
            [:div.columns
             [:div.column.is-narrow
              [:p.heading "longest running items"]
              (durations processed-metrics)]
             [:div.column network]]]]])))))


(defn generate-request-viz-file
  ([request-id]
   (spit "request-network.html" (generate-request-viz request-id)))
  ([metrics events]
   (spit "request-network.html" (generate-request-viz metrics events))))


(comment

  (require '[simply.helpers])

  (simply.helpers/with-db :pre-prod
    (generate-request-viz "8278e833-fc0f-4775-aa74-286e7a7a2cb5"))


  "PRE FETCH DATA"
  (def request-data
    (simply.helpers/with-db :emulator
      (let [request-id "41d0e8f0-91c8-4a11-9555-8d7e82a817c8"
            *metrics   (future (get-request-metrics request-id))
            *events    (future (get-request-events request-id))]
        {:metrics @*metrics
         :events @*events})))

  (generate-request-viz-file (:metrics request-data) (:events request-data))


  )
