(ns simply.auth.otp.core
  (:require [clj-time.core :as time]
            [clj-time.coerce :as time.coerce]
            [clojure.string :as string]
            [simply.auth.otp.db :as db]))


(defn- generate-otp [length]
  (let [max-random-digit 9]
    (string/join
     (take length
           (repeatedly #(rand-int max-random-digit))))))


(defn- otp-id [otp reference] (string/upper-case (str otp reference)))


(defn- new-otp
  [length max-age reference params]
  (let [otp (generate-otp length)]
    (assoc params
           :id (otp-id otp reference)
           :reference reference
           :otp otp
           :max-age max-age
           :requested-at (time.coerce/to-long (time/now)))))


(defn- expired? [{:keys [requested-at max-age] :as otp-entity}]
  (let [x-minutes-ago (time/plus (time/now) (time/minutes (- max-age)))]
    (time/before?
     (time.coerce/from-long requested-at)
     x-minutes-ago)))


(defn- correct? [otp otp-entity]
  (and
   (not (nil? otp-entity))
   ;chances of this happening is 0 unless someone tampers with db
   (= otp (:otp otp-entity))))


(defn request-otp
  "returns an otp string"
  [reference & {:keys [otp-length max-age-in-minutes params meta]
                :or   {otp-length          6
                       max-age-in-minutes 10
                       params {}
                       meta {}}}]
  {:pre [(not (string/blank? reference)) (map? params) (int? otp-length) (int? max-age-in-minutes)]}
  (let [otp-entity (new-otp otp-length max-age-in-minutes reference params)]
    (db/save otp-entity meta)
    (:otp otp-entity)))


(defn confirm-otp
  "if the otp is valid, returns the otp entity, else returns false"
  [otp reference]
  (let [id (otp-id otp reference)
        otp-entity (db/get-otp id)]
    (if-not (correct? otp otp-entity)
      false
      (do
        (db/delete id)
        (if (expired? otp-entity)
          false
          otp-entity)))))



(comment

  (def *last-otp (atom ""))
  (with-redefs [simply.deps/get-dep
                (constantly (simply.gcp.persistence.db/db
                             {:project-id "simply-pre-prod"
                              :client (simply.gcp.auth/keyfile-client
                                       (simply.gcp.keys/key-file :simply-pre-prod))}))]

   #_ (reset! *last-otp (request-otp "testA"
                                   :otp-length 10
                                   :max-age-in-minutes 1
                                   :params {:foo :foo/bar}
                                   :meta {:foo :keyword}))
   #_ (confirm-otp @*last-otp "testA"))

  )
