(ns simply.gcp.pubsub.cqrs
  (:require [simply.cqrs :refer [CQRS CqrsSystem] :as cqrs]
            [simply.core :as simply.core]
            [simply.gcp.pubsub.core :refer [Pubsub] :as p]
            [integrant.core :as ig]
            [simply.gcp.pubsub.encoding :as encoding]
            [clojure.spec.alpha :as s]
            [simply.errors :as e]
            [camel-snake-kebab.core :as camel-snake]))


(defn required-actions-topic-key [service]
  (str service "-required-actions"))


(defn required-actions-subscription-key [topic-key]
  (str topic-key "-sub"))


(defn required-actions-topic [topic-key]
  (p/topic topic-key encoding/encode-kebab-case-edn))


(defn- pubsub-cqrs [custom-event-wrapper pubsub service required-actions-topic-key]
  {:pre [(satisfies? Pubsub pubsub)]}
  (reify
    CQRS
    (require-actions [this actions]
      (p/publish pubsub required-actions-topic-key actions))

    (wrap-event [this event]
      (merge
       event
       {:sender service
        :date (simply.core/now-formatted)}
       (custom-event-wrapper event))
      )

    (notify [this topic coll]
      (p/publish pubsub topic coll))))


(defn- wrap-event-wrapper [event-wrapper]
  (fn [event]
    (let [event (update event :type camel-snake/->SCREAMING_SNAKE_CASE_STRING)]
      (if event-wrapper
        (event-wrapper event)
        event))))


(defn pubsub-cqrs-system
  [{:keys [client service events-topic event-wrapper topics subscriptions
           action-subscription-pull-interval-ms action-subscription-ack-deadline-in-sec] :as options}]
  {:pre [(string? service)
         (s/valid? :pubsub/topic events-topic)]}
  (let [topics (or topics [])
        *options (atom options)
        subscriptions (->> (or subscriptions [])
                           (map (fn [{:keys [topic key] :as sub}]
                                  (assoc sub :handler
                                         (fn [message]
                                           (cqrs/handle-topic topic key message @*options))))))
        action-handler (fn [action] (cqrs/action-handler @*options action))
        command-handler (fn [type data user] (cqrs/command-handler @*options type data user))
        action-topic-key (required-actions-topic-key service)
        action-topic (required-actions-topic action-topic-key)
        topics (-> topics
                   (conj action-topic)
                   (conj events-topic))
        action-subscription-key (required-actions-subscription-key action-topic-key)
        action-subscription-pull-interval-ms (if (int? action-subscription-pull-interval-ms)
                                               action-subscription-pull-interval-ms
                                               p/pull-interval-ms)
        action-subscription (cond-> (p/subscription action-subscription-key
                                                    action-topic-key
                                                    encoding/decode-kebab-case-edn
                                                    action-handler
                                                    action-subscription-pull-interval-ms)
                              (int? action-subscription-ack-deadline-in-sec)
                              (p/with-ack-deadline action-subscription-ack-deadline-in-sec))
        subscriptions (conj subscriptions action-subscription)
        pubsub (p/pubsub client topics subscriptions)
        cqrs (pubsub-cqrs (wrap-event-wrapper event-wrapper) pubsub service action-topic-key)
        system (reify
                 CqrsSystem
                 (get-command-handler [this] command-handler)
                 (get-action-handler [this] action-handler)
                 (send-messages [this topic coll]
                   (cqrs/notify cqrs topic coll))
                 (request-actions [this actions]
                   (cqrs/require-actions cqrs actions))
                 (start [this]
                   (p/init pubsub)
                   (p/subscribe-all pubsub))
                 (stop [this]
                   (p/unsubscribe-all pubsub)))]
    (swap! *options #(assoc %
                            :cqrs cqrs
                            :cqrs-system system))
    system))


;;;; TOPICS

(defmethod ig/init-key :simply.gcp.pubsub.cqrs/pubsub-cqrs-system
  [_ {:keys [dependencies] :as options}]
  (let [options (merge (dissoc options :dependecies)
                       dependencies)
        {:keys [start] :as system} (pubsub-cqrs-system options)]
    (cqrs/start system)
    system))


(defmethod ig/halt-key! :simply.gcp.pubsub.cqrs/pubsub-cqrs-system [_ system]
  (cqrs/stop system))


;;;; SUBSCRIPTIONS

(defn system-handles-topic-subscription []
  (e/throw-app-error
   "Oops, it seams like they system is not wired up correctly."))
