(ns simply.astronomix.macros
  (:require [clojure.string :as s]
            [simply.astronomix.astro :as astro]))


(def astro-js-module 'js/Astronomix)


(defn module-name->kebab-case
  "Converts module and sub module names from camel case to kebab case
   eg: DatePicker to date-picker or Menu.Item to menu-item"
  [input]
  (->> (re-seq #"\w[a-z0-9]*" input)
       (map s/lower-case)
       (s/join "-")))


(def get-symbol-name (comp symbol module-name->kebab-case))


(defn get-module-path [module-name]
  (s/split module-name #"\."))


(defn define-fn [fn-name]
  (let [fn-name (name fn-name)]
    `(defn ~(get-symbol-name fn-name) [& args#]
      (apply simply.astronomix.core/call-func ~fn-name args#))))


(defn define-prop [prop-name]
  (let [prop-name (name prop-name)]
    `(defn ~(get-symbol-name prop-name) [prop#]
      (simply.astronomix.core/get-prop ~prop-name prop#))))


(defn define-form-funcs [method-name]
  (let [method-name (name method-name)]
    `(defn ~(get-symbol-name method-name) [form# & args#]
       (apply simply.astronomix.core/call-js-func ((keyword ~method-name) form#) args#))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Functions for the different react libraries
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn define-reagent-component [component]
  (let [component (name component)]
    `(def ~(get-symbol-name component)
      (reagent.core/adapt-react-class
        (apply goog.object/getValueByKeys
          ~astro-js-module ~(get-module-path component))))))


(defn define-rum-component [component]
  (let [component (name component)]
    `(defn ~(get-symbol-name component) [& args#]
      (apply simply.astronomix.rum/adapt-class
        (apply goog.object/getValueByKeys
          ~astro-js-module ~(get-module-path component)) args#))))


(defmacro export-funcs []
  `(do ~@(map define-fn astro/funcs)))


(defmacro export-props []
  `(do ~@(map define-prop astro/props)))


(defmacro export-reagent-components []
  `(do ~@(map define-reagent-component astro/components)))


(defmacro export-rum-components []
  `(do ~@(map define-rum-component astro/components)))
