(ns simply.csv.generate
  (:require [clojure.string :as string]))


;; Replaces a double quotation mark with two double quotation marks
;; which will effectively escape the double quotation mark for CSV files.
(defn- escape-with-double-quotes [value]
  (string/replace value
                  #"\""
                  "\"\""))


(defn- quote-string [value]
  (str "\""
       (escape-with-double-quotes
        (or value ""))
       "\""))


(defn- ->csv-value [value]
  (quote-string value))


(defn- ->csv-values [values]
  (map ->csv-value
       values))


(defn- ->csv-row [values]
  (string/join ","
               (->csv-values values)))


(defn- ->csv-rows [rows]
  (string/join "\n" rows))


(defn ->csv-string
  "Generates a CSV string. A double quotation mark will be escaped
  using two double quotation marks as per the CSV RFC.

  Supported parameters:

  - `header-row` A vector or list of strings that represent the header row and column names of the CSV file.
  - `rows` A vector/list of vector/lists which represent rows. Each row will be seperated by a new line and each
          item will be seperated by a comma.

  Example:

  (->csv-string :header-row [\"Column one\", \"Column two\"]
                :rows [[\"one\" \"two\"]
                       [1 2]])"
  [& {:keys [header-row
             rows]}]
  (->csv-rows
   (into
    [(->csv-row header-row)]
    (map ->csv-row rows))))


(defn ->csv-file
  "Generates a CSV string and saves it to the given `file-path`. A double quotation mark will be escaped
  using two double quotation marks as per the CSV RFC.

  Supported parameters:

  - `file-path` The file path to save the csv string to.
  - `header-row` A vector or list of strings that represent the header row and column names of the CSV file.
  - `rows` A vector/list of vector/lists which represent rows. Each row will be seperated by a new line and each
           item will be seperated by a comma.

  Example:

  (->csv-file \"/Users/ghostdog/Desktop/test.csv\"
              :header-row [\"Column one\", \"Column two\"]
              :rows [[\"one\" \"two\"]
                     [1 2]])"
  [file-path & {:keys [header-row
                       rows]}]
  (spit file-path
        (->csv-string
         :header-row header-row
         :rows rows)))
