(ns simply.ops.tracing-db-reporter
  (:require [integrant.core :as ig]
            [manifold.bus :as m.bus]
            [manifold.stream :as m.stream]
            [simply.persistence.core :as p.db]
            [cheshire.core :as json]
            [hiccup.page :as hiccup]
            [taoensso.timbre :as logger]
            [simply.ops.tracing :as tracing]))


;; TO USE ADD to system.edn
;; :simply.ops.tracing-db-reporter/report-tracing-metrics-to-datastore
;; {:service "your-service"
;;  :report? #dyn/boolean #dyn/prop [SIMPLY_CORE_TRACING_TO_DATASTORE "true"]
;;  :simply.deps/deps #ig/ref :simply.deps/deps}


(defn- report-tracing-metric [service i]
  (try
    (p.db/upsert
      (p.db/entity
        :db-namespace (p.db/db-namespace "OPS")
        :entity-key "tracing-metrics"
        :id (str (:time i) "-" (:id i))
        :data (assoc i :service service)))
    (catch Exception e
      (logger/error (ex-info "Cannot persist instance metric" i e)))))


(defn- start-reporting-tracing-metrics-to-datastore! [service]
  (->> (m.bus/subscribe tracing/tracing-report-bus tracing/tracing-report-topic)
       (m.stream/consume #(report-tracing-metric service %))))


(defmethod ig/init-key :simply.ops.tracing-db-reporter/report-tracing-metrics-to-datastore [_ {:keys [service report?]}]
  (when report?
    (start-reporting-tracing-metrics-to-datastore! service)))


;;;; DATA

(defn- get-request-metrics [request-id]
  (->> (p.db/find-all
         (p.db/query
           :db-namespace (p.db/db-namespace "OPS")
           :entity-key "tracing-metrics"
           :params {:request-id request-id}))
       (map #(-> % ::p.db/data (assoc :db-id (::p.db/id %))))))


(defn- get-request-events [request-id]
  (->> (p.db/find-all
         (p.db/query
           :db-namespace (p.db/db-namespace "Events")
           :entity-key "event"
           :params {:requestId request-id}))
       (map #(-> % ::p.db/data (dissoc :payload)
                 (assoc :trigger "EVENT"
                        :db-id (::p.db/id %)
                        :id (::p.db/id %))))))

;;;; REQUEST METRICS

(defn- recursively-apply-metric-children
  ([metrics-by-parents ms] (recursively-apply-metric-children 1 metrics-by-parents ms))
  ([n metrics-by-parents ms]
   (if (empty? ms)
     ms
     (->> ms
          (map (fn [{:keys [id events] :as m}]
                 (let [children (get metrics-by-parents id [])
                       ne (inc n)
                       n' (if (empty? events) (inc n) (+ n 2))]
                   (-> m
                       (assoc :level n
                              :children (recursively-apply-metric-children n' metrics-by-parents children))
                       (update :events (fn [e] (map #(assoc % :level ne) e)))))))))))



(defn- unwind-trees [trees]
  (->> trees
       (map (fn [{:keys [events children] :as metric}]
              (->> [(dissoc metric :events :children)]
                   (into events)
                   (into (unwind-trees children)))))
       (reduce into)))


(defn- remove-required-actions [metrics]
  (let [metrics       (->> metrics
                           (map (fn [m]
                                  (assoc m :required-action?
                                         (and (clojure.string/starts-with? (:trigger m) "PUBSUB")
                                              (when-let [d (get-in m [:data :topic])]
                                                (some? (re-find #"required-action" d))))))))
        parent-lookup (into {} (map (juxt :id :parent-id) metrics))
        subs          (->> metrics
                           (filter #(and (= "PUBSUB_SUBSCRIPTION" (:trigger %))
                                         (:required-action? %)))
                           (map (fn [{:keys [id parent-id]}]
                                  [id (get parent-lookup parent-id)]))
                           (into {}))]
    (->> metrics
         (filter #(not (:required-action? %)))
         (map (fn [t]
                (if-let [parent (get subs (:parent-id t))]
                  (assoc t :parent-id parent)
                  t))))))


(defn- process-request-metrics [metrics events]
  (let [events-by-parent         (group-by :parent-id events)
          metrics-with-events      (->> metrics
                                        remove-required-actions
                                        (map (fn [{:keys [id] :as m}]
                                               (assoc m :events (get events-by-parent id []))))
                                        (sort-by :time))
          orphaned-events          (get events-by-parent nil [])
          metrics-by-parents       (group-by :parent-id metrics-with-events)
          top-level-metrics        (get metrics-by-parents nil [])
          trees                    (recursively-apply-metric-children metrics-by-parents top-level-metrics)

          trigger-frequencies      (->> (concat metrics-with-events events) (map :trigger) frequencies)]
      {:trigger-frequencies trigger-frequencies
       :metrics (unwind-trees trees)
       :top-level-metrics  top-level-metrics
       :orphaned-events     orphaned-events
       :trees               trees
       :duration            (- (:time (last metrics-with-events)) (:time (first metrics-with-events)))}))


;;;; VISUALIZE REQUEST

(defn- color [d]
  (case (:trigger d)
    "COMMAND"             "red"
    "EVENT"               "green"
    "ACTION"              "blue"
    "PUBSUB_SUBSCRIPTION" "orange"
    "PUBSUB_TOPIC"        "yellow"
    "grey"))


(defmulti ^:private node-opts :trigger)

(defmethod node-opts :default [d] {:label (:trigger d)})

(defmethod node-opts "COMMAND" [d]
  {:label (str "COMMAND\n" (clojure.string/lower-case (get-in d [:data :type])))})

(defmethod node-opts "EVENT" [d]
  {:label (str "EVENT\n\n" (clojure.string/lower-case (get-in d [:type])))})

(defmethod node-opts "ACTION" [d]
  {:label (str "ACTION\n\n" (clojure.string/lower-case (get-in d [:data :type])))})

(defmethod node-opts "PUBSUB_SUBSCRIPTION" [d]
  {:label (str "PULL\n\n" (clojure.string/lower-case (get-in d [:data :sub])))})

(defmethod node-opts "PUBSUB_TOPIC" [d]
  {:label (str "PUSH\n\n" (clojure.string/lower-case (get-in d [:data :topic])))})


(defn- processed-request-metrics-to-graph-network [processed-metrics]
  (let [{:keys [metrics]} processed-metrics
        nodes             (->> metrics
                               (map (fn [{:keys [id level service] :as e}]
                                      (-> (node-opts e)
                                          (assoc :id id :level level :color (color e))
                                          (update :label #(if service (str % "\n\n " service) %))))))
        edges             (->> metrics
                               (map (fn [{:keys [parent-id id] :as m}]
                                      (when parent-id
                                        {:from parent-id :to id :color (color m)})))
                               (remove nil?))]
    {:nodes nodes :edges edges}))


(defn- graph-component [{:keys [nodes edges] :as network}]
  (let [data-json    (json/generate-string network {:escape-non-ascii true})
        vis-opts     {:layout
                      {:hierarchical
                       {:enabled         true
                        :direction       "UD"
                        :sortMethod      "directed"
                        :levelSeparation 200}}
                      :edges {:arrows {:to {:enabled     true
                                            :type        "arrow"
                                            :scaleFactor 0.8}}
                              :smooth true}
                      :nodes {:shape           "box"
                              :widthConstraint {:maximum 67}
                              :margin          {:top 10 :bottom 10 :left 20 :right 20}}}
        options-json (json/generate-string vis-opts {:escape-non-ascii true})]
      [:div
       [:style "#network {height: 85vh}"]
       [:div#network]
       [:script {:type "text/javascript" :src "https://unpkg.com/vis-network/standalone/umd/vis-network.min.js"}]
       [:script {:type "text/javascript"}
        "var VIS_DATA = " data-json  ";"
        "var container = document.getElementById('network');"
        "var nodes = new vis.DataSet(VIS_DATA.nodes);"
        "var edges = new vis.DataSet(VIS_DATA.edges);"
        "var data = {nodes: nodes, edges: edges};"
        "var options = " options-json ";"
        "var network = new vis.Network(container, data, options);"]]))


(defn generate-request-viz
  ([request-id]
   (let [*metrics (future (get-request-metrics request-id))
         *events  (future (get-request-events request-id))]
     (generate-request-viz @*metrics @*events)))
  ([metrics events]
   (let [network  (->> (process-request-metrics metrics events)
                       processed-request-metrics-to-graph-network
                       graph-component)]
     (spit "request-network.html"
           (hiccup/html5
             {}
             [:html
              [:body
               network]])))))


(comment

  (require '[simply.helpers])

  (simply.helpers/with-db :pre-prod
    (generate-request-viz "8278e833-fc0f-4775-aa74-286e7a7a2cb5"))


  "PRE FETCH DATA"
  (def request-data
    (simply.helpers/with-db :pre-prod
      (let [request-id "8278e833-fc0f-4775-aa74-286e7a7a2cb5"
            *metrics   (future (get-request-metrics request-id))
            *events    (future (get-request-events request-id))]
        {:metrics @*metrics
         :events @*events})))

  (generate-request-viz (:metrics request-data) (:events request-data))


  )
