(ns simply.scheduling.db
  (:require [simply.scheduling.core :refer [ScheduleDb]]
            [simply.gcp.datastore.core :as api]
            [clj-time.core :as time]
            [clj-time.coerce :as time.coerce]
            [integrant.core :as ig]))


;;;; DATASTORE

(defn datastore-schedule-db
  "Uses google cloud datastore as the schedule DB"
  [{:keys [client project-id namespace] :as options}]
  (let [db-client (api/api-client options)
        jobs-entity "jobs"
        wrap-entity (api/wrap-entity project-id namespace)]
    (reify
      ScheduleDb

      (should-run-interval-job? [this job-key interval]
        (let [now (time/now)
              delay (time/plus now (time/millis (- interval)))
              past-delay? (fn [date] (-> date
                                         time.coerce/from-date
                                         (time/before? delay)))
              last-run (time.coerce/to-date now)
              transaction (api/start-transaction db-client)
              entity-key (api/name-key jobs-entity job-key)
              request-entity (-> entity-key
                                 (assoc :transaction transaction)
                                 wrap-entity)
              entity (api/lookup-entity db-client request-entity)
              exists? (not (nil? entity))
              should-run? (or (not exists?)
                              (past-delay? (:last-run entity)))
              entity-to-save (-> entity-key
                                 (assoc :job-name job-key
                                        :last-run last-run)
                                 wrap-entity)
              save-request (api/commit-request {:transaction transaction
                                                :mode api/transactional-mode
                                                :mutations [(api/upsert-mutation entity-to-save)]})]
          (if should-run?
            (try
              (db-client save-request)
              true
              (catch Exception e ;; we swallow the exception because the transaction failed
                false))
            false))))))


(defmethod ig/init-key :simply.scheduling.db/datastore-schedule-db
  [_ {:keys [client project-id namespace] :as options}]
  (datastore-schedule-db options))


;;;; DEV


(defn never-run-schedule-db
  "Jobs will never execute"
  []
  (reify
    ScheduleDb
    (should-run-interval-job? [_ _ _] false)))


(defmethod ig/init-key :simply.scheduling.db/never-run-schedule-db
  [_ _]
  (never-run-schedule-db))


(defn in-memory-schedule-db
  "Only for local development or when only one server will ever run."
  []
  (let [*jobs (atom {})]
    (reify
      ScheduleDb

      (should-run-interval-job? [this job-key interval]
        (let [now (time/now)
              last-expected-run (time/minus now (time/millis interval))
              should-run?
              (if-let [last-run (get @*jobs job-key)]
                (time/before? last-run last-expected-run)
                true)]
          (if should-run?
            (do (swap! *jobs #(assoc % job-key now))
                true)
            false))))))


(defmethod ig/init-key :simply.scheduling.db/in-memory-schedule-db
  [_ _]
  (in-memory-schedule-db))


(comment
  (let [job (simply.scheduling.core/interval-job "my job"
                                                 10000
                                                 #(prn "** WHOOP **"))
        stop-job (simply.scheduling.core/start-interval-job (in-memory-schedule-db) job)]
    (prn "Job will start in 10 seconds and run for 25 seconds")
    (Thread/sleep 35000)
    (stop-job)
    (prn "Job Stopped"))
  )
