(ns simply.persistence.core
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as string]
            [simply.errors :as e]
            [simply.deps :as deps]
            [simply.persistence.db :refer [Db] :as db]
            [integrant.core :as ig]))

;;;; ENTITIES

(def allowed-meta-annotations #{:keyword :timestamp})

(s/def ::string #(and (string? %) (not (string/blank? %))))
(s/def ::db-namespace (s/coll-of ::string :kind #(and (vector? %) (odd? (count %)))))
(s/def ::entity-key ::string)
(s/def ::id ::string)
(s/def ::data map?)
(s/def ::meta (s/map-of (s/coll-of keyword? :kind vector? :min-count 1) allowed-meta-annotations))
(s/def ::entity (s/keys :req [::db-namespace
                              ::entity-key
                              ::id
                              ::data
                              ::meta]))


(defn- guard-against-invalid-entity [entity]
  (when-let [err (s/explain-data ::entity entity)]
    (e/throw-app-error "Persistence entity does not conform to spec"
                       (dissoc err :clojure.spec.alpha/value)))
  (when (> (count (::db-namespace entity)) 1)
    (e/throw-app-error "Ancestor keys are not implemented yet"
                       (dissoc entity ::data))))


(defn db-namespace [& paths]
  (vec paths))


(defn entity
  [& {:keys [db-namespace entity-key id data meta]
      :or {db-namespace []
           entity-key ""
           id ""
           data {}
           meta {}}}]
  {::db-namespace db-namespace
   ::entity-key entity-key
   ::id id
   ::data data
   ::meta (->> meta
               (map (fn [[k v]] [(if (keyword? k) [k] k) v]))
               (into {}))})


;;;; QUERIES

(s/def ::params (s/map-of keyword? (s/or :string string?
                                         :number number?
                                         :boolean boolean?)))
(s/def ::limit pos-int?)

(s/def ::query (s/keys :req [::db-namespace
                             ::entity-key
                             ::params]
                       :opt [::limit]))

(defn query "a simple query that matches on and equality only"
  [& {:keys [db-namespace entity-key params limit]
      :or {db-namespace []
           entity-key ""
           params {}}}]
  (cond-> {::db-namespace db-namespace
           ::entity-key entity-key
           ::params params}
    (number? limit) (assoc ::limit limit)))


(defn- guard-against-invalid-query [query]
  (when-let [err (s/explain-data ::query query)]
    (e/throw-app-error "Persistence query does not conform to spec" err)))


;;;; API

(defn- ->db [] (deps/get-dep :simply.persistence.core/db :satisfies Db))


(defn upsert "inserts or updates the entity at id"
  [entity]
  (guard-against-invalid-entity entity)
  (db/upsert-entity (->db) entity))


(defn lookup "Finds a single entity by it's id"
  [entity]
  (guard-against-invalid-entity entity)
  (db/lookup-entity (->db) entity))


(defn find-all "finds all entities that match query"
  [query]
  (guard-against-invalid-query query)
  (db/query-entity (->db) query))


(defn delete [entity]
  (guard-against-invalid-entity entity)
  (db/delete-entity (->db) entity))



(comment

  (guard-against-invalid-entity
   (entity :db-namespace (db-namespace "SME")
           :entity-key "company-view"
           :id "Simply"
           :meta {[:foo] ::timestamp}))

  )
