(ns simply.ops.tracing-db-reporter
  (:require [integrant.core :as ig]
            [manifold.bus :as m.bus]
            [manifold.stream :as m.stream]
            [simply.persistence.core :as p.db]
            [taoensso.timbre :as logger]
            [simply.ops.tracing :as tracing]))


;; TO USE ADD to system.edn
;; :simply.ops.tracing-db-reporter/report-tracing-metrics-to-datastore
;; {:service "your-service"
;;  :report? #dyn/boolean #dyn/prop [SIMPLY_CORE_TRACING_TO_DATASTORE "true"]
;;  :simply.deps/deps #ig/ref :simply.deps/deps}


(defn- report-tracing-metric [service i]
  (try
    (p.db/upsert
      (p.db/entity
        :db-namespace (p.db/db-namespace "OPS")
        :entity-key "tracing-metrics"
        :id (str (:time i) "-" (:id i))
        :data (assoc i :service service)))
    (catch Exception e
      (logger/error (ex-info "Cannot persist instance metric" i e)))))


(defn- start-reporting-tracing-metrics-to-datastore! [service]
  (->> (m.bus/subscribe tracing/tracing-report-bus tracing/tracing-report-topic)
       (m.stream/consume #(report-tracing-metric service %))))


(defmethod ig/init-key :simply.ops.tracing-db-reporter/report-tracing-metrics-to-datastore [_ {:keys [service report?]}]
  (when report?
    (start-reporting-tracing-metrics-to-datastore! service)))



(defn- get-request-metrics [request-id]
  (->> (p.db/find-all
         (p.db/query
           :db-namespace (p.db/db-namespace "OPS")
           :entity-key "tracing-metrics"
           :params {:request-id request-id}))
       (map #(-> % ::p.db/data (assoc :db-id (::p.db/id %))))))


(defn- get-request-events [request-id]
  (->> (p.db/find-all
         (p.db/query
           :db-namespace (p.db/db-namespace "Events")
           :entity-key "event"
           :params {:requestId request-id}))
       (map #(-> % ::p.db/data (dissoc :payload)
                 (assoc :trigger "EVENT"
                        :db-id (::p.db/id %))))))


(defn- recursively-apply-metric-children [metrics-by-parents ms]
  (if (empty? ms)
    ms
    (->> ms
         (map (fn [{:keys [id] :as m}]
                (let [children (get metrics-by-parents id [])]
                  (assoc m :children (recursively-apply-metric-children metrics-by-parents children))))))))


(comment

  (require '[simply.helpers])


  (def data
    (let [request-id "4c8e7b51-81be-4ae0-af74-396cc491bb6d"]
      (simply.helpers/with-db :pre-prod
        {:metrics (get-request-metrics request-id)
         :events (get-request-events request-id)})))


  (let [{:keys [metrics events]} data
        events-by-parent (group-by :parent-id events)
        metrics-with-events (->> metrics
                                 (map (fn [{:keys [id] :as m}]
                                        (assoc m :events (get events-by-parent id []))))
                                 (sort-by :time)
                                 ;; provide a dept level
                                 (reduce (fn [{:keys [n r]} i]
                                           (let [n' (inc n)]
                                             {:r (conj r (assoc i :level n'))
                                              :n (+ n' (count (:events i)))}))
                                         {:n 0 :r []})
                                 :r)
        orphaned-events (get events-by-parent nil [])
        metrics-by-parents (group-by :parent-id metrics-with-events)
        top-level-metrics (get metrics-by-parents nil [])
        trees (recursively-apply-metric-children metrics-by-parents top-level-metrics)
        trigger-frequencies (->> (concat metrics-with-events events) (map :trigger) frequencies)]
    {:trigger-frequencies trigger-frequencies
     :metrics-with-events metrics-with-events
     :top-loevel-metrics top-level-metrics
     :orphaned-events orphaned-events
     :trees trees
     :duration (- (:time (last metrics-with-events)) (:time (first metrics-with-events)))}

    )

  )
