(ns simply.ops.tracing
  (:require [simply.threads]
            [manifold.bus :as m.bus]))


(def tracing-report-bus (m.bus/event-bus))
(def tracing-report-topic "instance-reports")

(defn report-tracing [i]
  (future
    (m.bus/publish! tracing-report-bus tracing-report-topic i)))


(defn get-property [k] (:value (simply.threads/get-property k)))
(defn set-property! [k v] (simply.threads/set-property! k v))
(defn clear-property! [p] (simply.threads/clear-property! p))

(defn clear-property-if-exists! [k]
  (when-let [p (simply.threads/get-property k)]
    (simply.threads/clear-property! p)))


(defn current-tracing-id [] (get-property :tracingid))
(defn current-request-id [] (get-property :requestid))
(defn current-user-id [] (get-property :userid))

(defn set-request-id [x] (set-property! :requestid x))
(defn set-user-id [x] (set-property! :userid x))
(defn set-parent-id [x] (set-property! :parentid x))

(defn clear-request-id-if-exists! []
  (clear-property-if-exists! :requestid))

(defn clear-user-id-if-exists! []
  (clear-property-if-exists! :userid))

(defn clear-parent-id-if-exists! []
  (clear-property-if-exists! :parentid))



(defmacro with-tracing
  [trigger trigger-data & body]
  `(let [prev-id# (get-property :tracingid)
         id# (str (java.util.UUID/randomUUID))
         tracing-id-p# (set-property! :tracingid id#)
         parent-id# (get-property :parentid)
         parent-id-p# (when prev-id# (set-property! :parentid prev-id#))
         time# (System/currentTimeMillis)
         start# (System/nanoTime)]
     (try
       (let [result# (do ~@body)
             end# (System/nanoTime)
             tracing-parent-id# (get-property :parentid)
             tracing-request-id# (get-property :requestid)
             tracing-user-id# (get-property :userid)
             duration# (- end# start#)
             trace-data# {:trigger    ~trigger
                          :id         id#
                          :data       ~trigger-data
                          :time       time#
                          :duration   duration# ; Devide by 1000000 for ms
                          :parent-id  tracing-parent-id#
                          :request-id tracing-request-id#
                          :user-id    tracing-user-id#}]
         (report-tracing trace-data#)
         result#)
       (finally
         (if prev-id#
           (set-property! :tracingid prev-id#)
           (clear-property! tracing-id-p#))
         (when parent-id-p#
           (clear-property! parent-id-p#))
         (when parent-id#
           (set-property! :parentid parent-id#))))))



(defmacro with-request-id
  "Checks if a requestid is defined. If not, set one and clear it after command has been handled"
  [& body]
  `(let [r# (get-property :requestid)]
     (if r#
       (do ~@body) ;someone else controls the request
       (let [id#   (str (java.util.UUID/randomUUID))
             prop# (set-property! :requestid id#)]
         (try
           ~@body
           (finally
             (clear-property! prop#)))))))


(defmacro with-request-user
  [current-user & body]
  `(let [previous-user# (get-property :userid)
         prop#          (set-property! :userid (:user-id ~current-user))]
     (try
       ~@body
       (finally
         (clear-property! prop#)
         (when-not (empty? previous-user#)
           (set-property! :userid previous-user#))))))


(comment

  (macroexpand-1 '(with-tracing "FOO" {:bar 1} (+ 1 1)))
  )
