(ns simply.cqrs
  (:require [clojure.spec.alpha :as s]
            [expound.alpha :as expound]
            [taoensso.timbre :as log]
            [simply.errors :as e]
            [slingshot.slingshot :as slingshot]
            [simply.ops.tracing]))


(defprotocol CQRS
  ;; Queues Up Domain Actions
  (require-actions [this actions])
  ;; Wrap an Event With Additional Data
  (wrap-event [this event])
  ;; Push a collection of messages onto a publish subscribe topic
  (notify [this topic coll]))


(defprotocol CqrsSystem
  (get-command-handler [this])
  (get-action-handler [this])
  (send-messages [this topic coll])
  (request-actions [this actions])
  (start [this])
  (stop [this]))



(defmulti handle-command
  "Params:
   * Type is a keyword
   * Data is any data required by the handler
   * Options includes any dependencies"
  (fn [type data options] type))


(defmulti handle-action (fn [type data options] type))


(defmulti handle-topic (fn [topic-key subscription-key data options]
                         [topic-key subscription-key]))


;;;; EVENTS

(s/def :cqrs/type keyword?)
(s/def :cqrs/payload map?)
(s/def :cqrs/event (s/keys :req-un [:cqrs/type]
                           :opt-un [:cqrs/payload]))
(s/def :cqrs/events (s/coll-of :cqrs/event :kind vector :distinct true))


(defn event [type payload]
  {:post [(s/valid? :cqrs/event %)]}
  {:type type :payload payload})


;;;; ACTIONS

(defn action-has-valid-type? [{:keys [type]}]
  (contains? (-> handle-action methods keys set) type))


(s/def :cqrs/action-type action-has-valid-type?)
(s/def :cqrs/data some?)
(s/def :cqrs/action (s/and
                     (s/keys :req-un [:cqrs/type
                                      :cqrs/data])
                     :cqrs/action-type))
(s/def :cqrs/actions (s/coll-of :cqrs/action :kind vector))


(defn action [type data]
  {:post [(s/valid? :cqrs/action %)]}
  {:type type :data data})


(defn- with-user [m user]
  (assoc m :user user))


;;;; CHUNKS

(def max-chunk-total 500)


(defn chunks [size coll]
  (let [parts (partition size size nil coll)
        total-parts (count parts)
        should-chunk? (> total-parts 1)
        should-half-chunk? (> total-parts max-chunk-total)
        half-size (int (/ (count coll) 2))
        final-parts (if should-half-chunk?
                      (partition half-size half-size nil coll)
                      parts)]
    {:should-chunk? should-chunk?
     :parts final-parts}))


(defn requeue-action-chunks-if-needed
  [size cqrs type coll f]
  "This function is used in conjunction with handle-action.
   It allows an action to have a collection as it's data. The collection items can then be broken into smaller chunks that are in return handled by the system.
   This is mainly due to constraints on systems like pubsub and datastore that can only store x entities or send x messages as a batch.
  * Size is and int representing the largest allowed chunk
  * cqrs is an instance of CQRS
  * type is an action type
  * coll is the action data collection
  * f is the function that operates on the smallest allowed chunk"
  {:pre [(int? size) (satisfies? CQRS type) (s/valid? :cqrs/type type)]}
  (when-not (empty? coll)
    (let [{:keys [should-chunk? parts]} (chunks size coll)]
      (if-not should-chunk?
        (f)
        (do
          (log/infof "Chunking => %s actions into %s requests" type (count parts))
          (->> parts
               (map #(action type %))
               (require-actions cqrs)))))))


(defn chunk-events [cqrs type coll f]
  (requeue-action-chunks-if-needed 500 cqrs type coll f))


(defn chunk-entities [cqrs type coll f]
  (requeue-action-chunks-if-needed 20 cqrs type coll f))


(defn chunk-one-at-a-time [cqrs type coll f]
  (requeue-action-chunks-if-needed 1 cqrs type coll f))


;;;; CQRS - COMMANDS

(defn command-result
  ([events] (command-result events [] {}))
  ([events actions] (command-result events actions {}))
  ([events actions data]
   {:events events :actions actions :data data}))


(defn add-event-results
  "Appends extra events onto an existing command result"
  [command-result & events]
  (update command-result :events into (remove nil? events)))


(defn add-action-results
  "Appends extra actions onto an existing command result"
  [command-result & actions]
  (update command-result :actions into (remove nil? actions)))


(defn- event-actions [cqrs user events]
  (->> events
       (map #(wrap-event cqrs %))
       (map #(assoc %
                    :user user
                    :user-id (:user-id user)
                    :request-id (simply.ops.tracing/current-request-id)
                    :parent-id (simply.ops.tracing/current-tracing-id)))
       (action :enqueue-events)))


(defn command-handler
  "Handles a command and routes the resulting events and actions
   Params:
   * options: an options map is passed to implemented command handlers.
              should contain a :cqrs key that implements CQRS Protocol
   * type: a keyword representing the handle-command multimethod type
   * data: data required by the handle-command multimethod type"
  [{:keys [cqrs] :as options} type data user]
  {:pre [(s/valid? :cqrs/type type) (satisfies? CQRS cqrs)]}
  (simply.ops.tracing/with-request-id
    (simply.ops.tracing/with-request-user user
      (simply.ops.tracing/with-tracing "COMMAND" {:type (str type)}
        (let [{:keys [events actions data]} (handle-command type data (dissoc options :cqrs))
              event-errors  (or (s/explain-data :cqrs/events events)
                                (when (empty? events) "At least one event required"))
              action-errors (s/explain-data :cqrs/actions actions)]
          (when (or event-errors action-errors)
            (slingshot/throw+ (e/app-error "Invalid Command Result"
                                           {:event-errors
                                            (expound/expound-str :cqrs/events events)
                                            :action-errors
                                            (expound/expound-str :cqrs/actions actions)})))
          (let [events-action (event-actions cqrs user events)
                all-actions (->> (conj actions events-action)
                                 (map #(with-user % user)))]
            (require-actions cqrs all-actions)
            data))))))


;;;; CQRS - ACTIONS

(s/def :cqrs/action-result
  (s/keys :req-un [:cqrs/events]))


(defn action-result
  ([] (action-result []))
  ([events] {:events events}))


(defn action-handler
  [{:keys [cqrs] :as options} {:keys [type data user] :as action}]
  {:pre [(satisfies? CQRS cqrs)]}
  (log/infof "Action => %s %s "
             type (if (or (vector? data) (list? data) (seq? data)) (count data) ""))
  (simply.ops.tracing/with-request-id
    (simply.ops.tracing/with-tracing "ACTION" {:type (str type)}
      (let [{:keys [events]} (handle-action type data options)
            event-errors     (when events (s/explain-data :cqrs/events events))]
        (when event-errors
          (slingshot/throw+ (e/app-error "Invalid Action Result"
                                         {:event-errors
                                          (expound/expound-str :cqrs/events events)})))
        (when-not (empty? events)
          (require-actions cqrs [(event-actions cqrs user events)]))))))


;;;; USERS

(defn cqrs-user
  ([user-id] (cqrs-user user-id {}))
  ([user-id user-data]
   (merge user-data {:user-id user-id})))


(defn cqrs-thread-user []
  (cqrs-user (or (simply.ops.tracing/current-user-id) "unknown")))

;;;; CQRS - EVENTS

(def event-topic "events")

(defmethod handle-action :enqueue-events [type events {:keys [cqrs]}]
  (chunk-events cqrs type events
                (fn []
                  (->> events
                       (notify cqrs event-topic)))))



;;;; TOPIC SUBSCRIPTIONS

(defmethod handle-topic :default [t s _ _]
  (e/throw-app-error (str "You have no handle-topic defmethod defined for topic: " t " and subscription" s)))
