(ns simply.ops.tracing
  (:require [simply.threads]
            [manifold.bus :as m.bus]))


(def tracing-report-bus (m.bus/event-bus))
(def tracing-report-topic "instance-reports")

(defn report-tracing [i]
  (future
    (m.bus/publish! tracing-report-bus tracing-report-topic i)))


(defn current-tracing-id [] (:value (simply.threads/get-property :tracingid)))


(defmacro with-tracing
  [trigger trigger-data & body]
  `(let [prev-id# (:value (simply.threads/get-property :tracingid))
         id# (str (java.util.UUID/randomUUID))
         tracing-id-p# (simply.threads/set-property! :tracingid id#)
         parent-id# (:value (simply.threads/get-property :parentid))
         parent-id-p# (when prev-id# (simply.threads/set-property! :parentid prev-id#))
         time# (System/currentTimeMillis)
         start# (System/nanoTime)]
     (try
       (let [result# (do ~@body)
             end# (System/nanoTime)
             tracing-parent-id# (:value (simply.threads/get-property :parentid))
             tracing-request-id# (:value (simply.threads/get-property :requestid))
             tracing-user-id# (:value (simply.threads/get-property :userid))
             duration# (- end# start#)
             trace-data# {:trigger    ~trigger
                          :id         id#
                          :data       ~trigger-data
                          :time       time#
                          :duration   duration# ; Devide by 1000000 for ms
                          :parent-id  tracing-parent-id#
                          :request-id tracing-request-id#
                          :user-id    tracing-user-id#}]
         (report-tracing trace-data#)
         result#)
       (finally
         (if prev-id#
           (simply.threads/set-property! :tracingid prev-id#)
           (simply.threads/clear-property! tracing-id-p#))
         (when parent-id-p#
           (simply.threads/clear-property! parent-id-p#))
         (when parent-id#
           (simply.threads/set-property! :parentid parent-id#))))))
