(ns simply.ops.tracing
  (:require [simply.threads]
            [manifold.bus :as m.bus]))


(def tracing-report-bus (m.bus/event-bus))
(def tracing-report-topic "instance-reports")

(defn report-tracing [i]
  (m.bus/publish! tracing-report-bus tracing-report-topic i))


(defn current-tracing-id [] (:value (simply.threads/get-property :tracingid)))


(defmacro with-tracing
  [trigger trigger-data & body]
  `(let [prev-id# (:value (simply.threads/get-property :tracingid))
         id# (str (java.util.UUID/randomUUID))
         tracing-id-p# (simply.threads/set-property! :tracingid id#)
         parent-id# (:value (simply.threads/get-property :parentid))
         parent-id-p# (when prev-id# (simply.threads/set-property! :parentid prev-id#))
         time# (System/currentTimeMillis)
         start# (System/nanoTime)]
     (try
       (let [result# (do ~@body)
             end# (System/nanoTime)
             thread-parent-id (:value (simply.threads/get-property :parentid))
             thread-request-id (:value (simply.threads/get-property :requestid))
             thread-user-id (:value (simply.threads/get-property :userid))]
         (future
           (let [duration# (- end# start#)]
             (simply.ops.tracing/report-tracing {:trigger    ~trigger
                                                 :id         id#
                                                 :data       ~trigger-data
                                                 :time       time#
                                                 :duration   duration# ; Devide by 1000000 for ms
                                                 :parent-id  thread-parent-id
                                                 :request-id thread-request-id
                                                 :user-id    thread-user-id})))
         result#)
       (finally
         (if prev-id#
           (simply.threads/set-property! :tracingid prev-id#)
           (simply.threads/clear-property! tracing-id-p#))
         (when parent-id-p#
           (simply.threads/clear-property! parent-id-p#))
         (when parent-id#
           (simply.threads/set-property! :parentid parent-id#))))))
