(ns simply.auth.otp.core
  (:require [clj-time.core :as time]
            [clj-time.coerce :as time.coerce]
            [clojure.string :as string]
            [simply.auth.otp.db :as db]))


(defn- generate-otp [length]
  (let [max-random-digit 9]
    (string/join
     (take length
           (repeatedly #(rand-int max-random-digit))))))


(defn- otp-id [reference] (string/upper-case reference))


(defn- new-otp
  [length max-age max-tries reference params]
  (let [otp (generate-otp length)]
    (assoc params
           :id (otp-id reference)
           :reference reference
           :otp otp
           :max-age max-age
           :tries-left max-tries
           :requested-at (time.coerce/to-long (time/now)))))


(defn- time-limited? [{:keys [requested-at max-age] :as otp-entity}]
  (let [x-minutes-ago (time/plus (time/now) (time/minutes (- max-age)))]
    (time/before?
     (time.coerce/from-long requested-at)
     x-minutes-ago)))


(defn- try-limited? [{:keys [tries-left] :as otp-entity}]
  (<= tries-left 0))


(defn- expired? [otp-entity]
  (or (try-limited? otp-entity) (time-limited? otp-entity)))


(defn- not-matching? [otp otp-entity]
  (not= otp (:otp otp-entity)))


(defn request-otp
  "returns an otp string"
  [reference & {:keys [otp-length max-age-in-minutes params meta max-tries]
                :or   {otp-length         6
                       max-age-in-minutes 10
                       max-tries          3
                       params             {}
                       meta               {}}}]
  {:pre [(not (string/blank? reference)) (map? params) (int? otp-length) (int? max-age-in-minutes) (int? max-tries)]}
  (let [otp-entity (new-otp otp-length max-age-in-minutes max-tries reference params)]
    (db/save otp-entity meta)
    (:otp otp-entity)))


(defn confirm-otp
  "if the otp is valid, returns the otp entity, else returns false"
  [otp reference & {:keys [meta]
                    :or {meta {}}}]
  (let [id (otp-id reference)
        otp-entity (db/get-otp id)]
    (cond
      (nil? otp-entity)
      false

      (expired? otp-entity)
      (do
        (db/delete id)
        false)

      (not-matching? otp otp-entity)
      (do
        (db/save (update otp-entity :tries-left dec) meta)
        false)

      :else
      (do
        (db/delete id)
        otp-entity))
    ))



(comment

  (def *last-otp (atom ""))
  (with-redefs [simply.deps/get-dep
                (constantly (simply.gcp.persistence.db/db
                             {:project-id "simply-pre-prod"
                              :client (simply.gcp.auth/keyfile-client
                                       (simply.gcp.keys/key-file :simply-pre-prod))}))]

    (reset! *last-otp (request-otp "testA"
                                   :otp-length 10
                                   :max-tries 2
                                   :max-age-in-minutes 10
                                   :params {:foo :foo/bar}
                                   :meta {:foo :keyword}))
    (confirm-otp @*last-otp "testA"))

  )
