(ns simply.job-requests.core
  (:require [simply.cqrs :as cqrs]
            [integrant.core :as ig]
            [simply.deps :as s.deps]
            [simply.job-requests.db :as j.db]
            [simply.ops.tracing]))

;; SCHEDULING

(defn- db [] (s.deps/get-dep :simply.job-requests.core/db))

(defn job [jobname cron-schedule service-name] {:name jobname :schedule cron-schedule :service service-name})

(defn list-scheduled-jobs [] (j.db/list-jobs (db)))

(defn schedule-job [job] (j.db/create (db) job))

(defn de-schedule-job [job] (j.db/delete (db) job))


;; HANDLING

(def job-request-topic-key "simply-cron-jobs")

(defmulti handle-job-request (fn [data options] (:jobname data)))

(defn registered-handlers [] (set (keys (methods simply.job-requests.core/handle-job-request))))

(defn job-request-subscription-key [service-name]
  (str service-name "-" job-request-topic-key "-sub"))


(defmethod cqrs/handle-command :job-started
  [_ data _]
  (cqrs/command-result [(cqrs/event :auto-job-started data)]))

(defmethod cqrs/handle-command :job-finished
  [_ data _]
  (cqrs/command-result [(cqrs/event :auto-job-finished data)]))


(defn subscribe-to-cron-job-requests!
  "Creates a topic subscription that listens for job requests targeted at this service and calls handle-job-request if the service matches"
  [service-name]
  (let [sub-key (job-request-subscription-key service-name)]
    (defmethod cqrs/handle-topic
      [job-request-topic-key sub-key]
      [_ _ data options]
      (let [{:keys [service]} data]
        (when (= service service-name)
          (simply.ops.tracing/with-tracing "JOB" data
            (let [handle-command (cqrs/get-command-handler (:cqrs-system options))
                  event-data (assoc data :job-id (str (java.util.UUID/randomUUID)))
                  notify #(handle-command % event-data (cqrs/cqrs-user "simply.core.jobhandler"))]
              (notify :job-started)
              (handle-job-request data options)
              (notify :job-finished))))))))


(defmethod ig/init-key :simply.job-requests.core/register-job-handler
  [_ {:keys [service]}]
  (subscribe-to-cron-job-requests! service))


;;;; TESTING

(defmethod handle-job-request "test-job-request-queue"
  [data options]
  (prn "*** JOB RECEIVED WITH DATA ***")
  (prn data)
  (prn "*** AND OPTIONS ***")
  (prn (keys options)))
