(ns simply.ops
  (:require [shutdown.core :as shutdown]
            [simply.cqrs :as cqrs]
            [simply.messaging :as messaging]
            [integrant.core :as ig]
            [manifold.bus :as m.bus]
            [manifold.stream :as m.stream]
            [simply.persistence.core :as p.db]
            [taoensso.timbre :as logger]))


;; TO USE ADD THE FOLLOWING TO YOUR system.edn file as well as a add the slack topic

;:simply.ops/system-notifications
;{:version #dyn/prop [CI_VERSION "development"]
; :project-id #dyn/prop [GCP_PROJECT_ID "dogmatix"]
; :cqrs-system #ig/ref :suffix/cqrs-system-provider
; :slack-token "your token here"}

(def slack-room "system-ops")


(defn notify-startup [slack-token project-id version cqrs-system]
  (let [message (str "STARTING: *" version "* on *" project-id "*")
        slack (messaging/slack slack-token slack-room "system-startup" message)]
    (cqrs/send-messages cqrs-system messaging/slack-topic [slack])))


(defn notify-shutdown [slack-token project-id version cqrs-system]
  (let [message (str "HALT: *" version "* on *" project-id "*")
        slack (messaging/slack slack-token slack-room "system-shutdown" message)]
    (cqrs/send-messages cqrs-system messaging/slack-topic [slack])))


(defmethod ig/init-key :simply.ops/system-notifications [_ {:keys [slack-token cqrs-system project-id version]}]
  (let [shutdown #(notify-shutdown slack-token project-id version cqrs-system)]
    (notify-startup slack-token project-id version cqrs-system)
    (shutdown/add-hook! ::ops-notifications shutdown)))


(defmethod ig/halt-key! :simply.ops/system-notifications [_ _]
  (shutdown/remove-hook! ::ops-notifications))



;; TO USE ADD to system.edn
;; :simply.ops/report-tracing-metrics-to-datastore
;; {:service "your-service"
;;  :report? #dyn/boolean #dyn/prop [SIMPLY_CORE_TRACING_TO_DATASTORE "true"]}


(defn- report-tracing-metric [service i]
  (try
    (p.db/upsert
      (p.db/entity
        :db-namespace (p.db/db-namespace "OPS")
        :entity-key "tracing-metrics"
        :id (str (:time i) "-" (:id i))
        :data (assoc i :service service)))
    (catch Exception e
      (logger/error (ex-info "Cannot persist instance metric" i e)))))


(defn- start-reporting-tracing-metrics-to-datastore! [service]
  (->> (m.bus/subscribe cqrs/tracing-report-bus cqrs/tracing-report-topic)
       (m.stream/consume #(report-tracing-metric service %))))


(defmethod ig/init-key :simply.ops/report-tracing-metrics-to-datastore [_ {:keys [service report?]}]
  (when report?
    (start-reporting-tracing-metrics-to-datastore! service)))
(comment

  (require '[simply.helpers])


  (def request-id
    (->
      (simply.helpers/with-db :pre-prod
        (p.db/find-all
          (p.db/query
            :db-namespace (p.db/db-namespace "OPS")
            :entity-key "tracing-metrics"
            :limit 1)))
      first
      ::p.db/data
      :request-id))


  (def data
    (->>
      (concat
        (->>
          (simply.helpers/with-db :pre-prod
            (p.db/find-all
              (p.db/query
                :db-namespace (p.db/db-namespace "OPS")
                :entity-key "tracing-metrics"
                :params {:request-id request-id})))
          (map ::p.db/data))

        (->>
          (simply.helpers/with-db :pre-prod
            (p.db/find-all
              (p.db/query
                :db-namespace (p.db/db-namespace "Events")
                :entity-key "event"
                :params {:requestId request-id})))
          (map #(-> % ::p.db/data
                    (assoc :trigger "EVENT"
                           :time (.getTime (.parse (java.text.SimpleDateFormat. "yyyy-MM-dd'T'HH:mm:ss.SSS"  ) (get-in % [::p.db/data :date]))))))))))


  (let [children-by-parent (group-by :parent-id data)
        tree (loop [nodes (get children-by-parent nil [])]
               (prn nodes)
               (if (empty? nodes)
                 nodes
                 (->> nodes
                      (map (fn [n]
                             (-> (assoc n :children (get children-by-parent (:id n) []))
                                 (update n :children #(recur %)))))
                      doall)))]
    tree)


  )
