(ns simply.gcp.url
  (:require [clojure.string :as string]
            [clojure.data.json :as json]
            [simply.http.client :as http]
            [simply.core :refer [ShortenUrl] :as s.core]
            [integrant.core :as ig]
            [taoensso.timbre :as log]
            [cemerick.url :as url]))


;;;; URLS must contain `https://{optionally.anything.}simply.co.za{optionally anything}`
;;;; Use the following whitelisted regex: `https:\/\/?(.*)simply+\.co\.za?(.*)`


(defn- shorten-request [key short-domain long-url]
  {:method :post
   :url (format "https://firebasedynamiclinks.googleapis.com/v1/shortLinks?key=%s" key)
   :body (json/write-str {:longDynamicLink (str short-domain (url/url-encode long-url))
                          :suffix {:option "SHORT"}})
   :headers {"Content-Type" "application/json"}})


(defn- shorten [key short-domain long-url]
  (if key
    (let [request (shorten-request key short-domain long-url)
          response @(http/request request)
          valid? (= 200 (:status response))]
      (when-not valid?
        (log/error (ex-info "Could Not Shorten Url" (select-keys response [:body :status]))))
      (if valid?
        (-> response :body json/read-str (get "shortLink"))
        long-url))
    long-url))


(defn url-shortner [api-key]
  (reify
    ShortenUrl
    (try-shorten [this long-url]
      (shorten api-key "https://smply.page.link?link="  long-url))
    (try-shorten [this short-domain long-url]
      (shorten api-key short-domain long-url))))


(defmethod ig/init-key :simply.gcp.url/url-shortner
  [_ {:keys [api-key]}]
  (url-shortner api-key))


(comment
  ;;;; BAD KEY returns LONG URL and LOGS Error
  (s.core/try-shorten (url-shortner "123") "floop shortness" "http://sneakycode.net")

  ;;;; GOOD KEY will NOT shorten non simply
  (let [api-key ""]
    (s.core/try-shorten (url-shortner api-key) "http://sneakycode.net"))

  ;;;; GOOD KEY and simply Link will shorten
  (let [api-key ""]
    (s.core/try-shorten (url-shortner api-key) "https://simply.co.za"))


   ;;;; GOOD KEY and url params should work
  (let [api-key ""]
    (s.core/try-shorten (url-shortner api-key) "https://simply.co.za?foo=1&bar=1"))
  )
