(ns simply.job-requests.schedule
  (:require [simply.job-requests.core :as jr]
            [simply.deps :as s.deps]
            [clojure.data :as clojure.data]
            [taoensso.timbre :as log]
            [integrant.core :as ig]))

(defn- get-service [] (s.deps/get-dep :app/service-name))


(defn- service-job? [job] (= (:service job) (get-service)))


(defn- existing-jobs []
  (->> (jr/list-scheduled-jobs)
       (filter service-job?)))


(defn- manage-jobs [job-schedules]
  (let [service (get-service)
        existing-jobs (existing-jobs)
        existing-jobs-map (->> existing-jobs (map (juxt :name identity)) (into {}))
        jobs-to-add (->> job-schedules
                         (filter
                          (fn [[_ v]]
                            (not= "paused" v)))
                         (filter
                          (fn [[k _]]
                            (not (contains? existing-jobs-map k))))
                         (map (fn [[k s]] (jr/job k s service))))
        jobs-to-delete (->> existing-jobs
                            (filter #(not (contains? job-schedules (:name %)))))
        jobs-changed (->> existing-jobs
                          (map (fn [{:keys [name schedule] :as j}]
                                 (when-let [s (get job-schedules name)]
                                   (when (not= schedule s)
                                     [j (assoc j :schedule s)]))))
                          (remove nil?))]
    (doseq [[old new] jobs-changed]
      (jr/de-schedule-job old)
      (jr/schedule-job new))
    (doseq [job jobs-to-delete]
      (jr/de-schedule-job job))
    (doseq [job jobs-to-add]
      (jr/schedule-job job))))


(defn- cancel-all-jobs []
  (doseq [job (existing-jobs)]
    (jr/de-schedule-job job)))


(defn stats [job-schedules]
  (let [schedules (set (keys job-schedules))
        registered (jr/registered-handlers)
        diff (clojure.data/diff schedules registered)]
    {:active (last diff)
     :not-scheduled (disj (second diff) "test-job-request-queue")
     :not-registered (first diff)}))


(defn schedule-jobs
  "takes a map in the form {\"some-job-id\" \"* * * * *\"}"
  [run-jobs? job-schedules]
  (if (true? run-jobs?)
    (manage-jobs job-schedules)
    (cancel-all-jobs)))


(defn stats [job-schedules]
  (let [schedules (set (keys job-schedules))
        registered (jr/registered-handlers)
        diff (clojure.data/diff schedules registered)]
    {:active (last diff)
     :not-scheduled (disj (second diff) "test-job-request-queue")
     :not-registered (first diff)}))


(defn log-stats [job-schedules]
  (let [{:keys [not-scheduled not-registered]} (stats job-schedules)]
    (doseq [j not-scheduled]
      (log/error (str "JOB REGISTERED BUT NOT SCHEDULED: " j)))
    (doseq [j not-registered]
      (log/error (str "JOB SCHEDULED BUT NOT REGISTERED: " j)))))


(defmethod ig/init-key :simply.job-requests.schedule/jobs [_ {:keys [schedule run-jobs?]
                                                              :or {schedule  {}
                                                                   run-jobs? false}}]
  (log-stats schedule)
  (schedule-jobs run-jobs? schedule))


(comment
  (require 'simply.gcp.auth 'simply.gcp.keys
           'simply.gcp.cloud-scheduler 'simply.job-requests.cloud-scheduler)

  (simply.gcp.auth/reset-credentials!)

  (let [project-id "simply-pre-prod"
        client     (simply.gcp.auth/keyfile-client (simply.gcp.keys/key-file :simply-pre-prod))
        schedule   {"test-job-request-queue" "* * * * *"}
        run-jobs? false]
    (with-redefs [simply.deps/get-dep (fn [k]
                                        (case k
                                          :gcp/project-id              project-id
                                          :gcp/client                  client
                                          :app/service-name            "suffix-2"
                                          :simply.job-requests.core/db (simply.job-requests.cloud-scheduler/db)))]
      (schedule-jobs run-jobs? schedule)))
  )
