(ns simply.gcp.persistence.db
  (:require [simply.persistence.db :refer [Db] :as protocol]
            [simply.persistence.core :as p]
            [simply.persistence.ns :as p.ns]
            [simply.gcp.datastore.core :as api]
            [integrant.core :as ig]
            [simply.core :as c]
            [clojure.walk :as walk]
            [clojure.string :as string]
            [simply.gcp.keys :as ks]))


;;;; Meta Transforms
;;;; entity fields can be annotated with a "type"
;;;; we convert these data for these types either to
;;;; * make them easier to read in the db (example a date time vs a timestamp)
;;;; * handle types that the db does not support (example keywords)

(def ^:private *meta-transforms (atom {}))

(defn- register-transform [k & {:keys [to-db-entity from-db-entity]}]
  (swap! *meta-transforms assoc k {:to to-db-entity :from from-db-entity}))


(register-transform :keyword
                    :to-db-entity c/keyword->ns-str
                    :from-db-entity keyword)

(register-transform :timestamp
                    :to-db-entity #(java.util.Date. %)
                    :from-db-entity #(.getTime %))

(register-transform :set
                    :to-db-entity vec
                    :from-db-entity set)


(defn- apply-meta-transforms [direction data meta]
  (reduce
   (fn [d [p t]]
     (if-some [v (get-in d p)]
       (let [f (get-in @*meta-transforms [t direction])]
         (update-in d p f))
       d))
   data
   meta))


(defn- apply-to-db-meta-transforms [data meta]
  (apply-meta-transforms :to data meta))


(defn- apply-from-db-meta-tranforms [data meta]
  (apply-meta-transforms :from data meta))


(defn meta->db-meta [meta]
  (->> meta
       (map (fn [[k v]]
              {:path (map c/keyword->ns-str k) :type(c/keyword->ns-str v)}))))


(defn db-meta->meta [meta]
  (->> meta
       (map (fn [{:keys [path type]}]
              [(map keyword path) (keyword type)]))
       (into {})))


;;;; Keyword Transforms
;;;; The db does not know how to deal with keywords
;;;; and if the caller forgets to annotate the keyword we do it for them
;;;; by turning it into a string and appending _kw__
;;;; so we can turn it back into a keyword when we fetch the data

(def ^:private keyword-prefix "_kw__")
(def ^:private keyword-prefix-length (count keyword-prefix))

(defn- keyword->db-keyword [k]
  (str keyword-prefix (c/keyword->ns-str k)))

(defn- db-keyword->keyword [s]
  (if (string/starts-with? s keyword-prefix)
    (keyword (subs s keyword-prefix-length))
    s))

(defn- transform-keyword-values [test? transform data]
  (walk/prewalk
   (fn [n]
     (if (map? n)
       (->> n
            (map (fn [[k v]]
                   [k
                    (if (test? v)
                      (transform v)
                      v)]))
            (into {}))
       n))
   data))


(defn- transform-keywords-to-db-keywords [data]
  (transform-keyword-values keyword? keyword->db-keyword data))

(defn- transform-db-keywords-to-keywords [data]
  (transform-keyword-values string? db-keyword->keyword data))


;;;; Entity -> Db Entity
;;;; * Transform data based on annotations
;;;; * Transform keywords that are not annotated
;;;; * Index all primitives on the main map
;;;; * Included meta in db entity

(defn entity->namespace [entity]
  (first (::p/db-namespace entity)))


(defn- entity->identity [entity]
  (api/name-key (::p/entity-key entity) (::p/id entity)))


(def ^:private index-tests #{keyword? string? inst? number? boolean?})

(defn- data->indexes [data]
  (->> data
       (filter (fn [[_ v]] (some #(% v) index-tests)))
       (map first)
       set))


(defn auto-detect-meta [entity]
  (let [auto-meta (->> (::p/data entity)
                       (filter (fn [[k v]] (set? v)))
                       (map (fn [[k _]]
                              [[k] :set]))
                       (into {}))]
    (update entity ::p/meta (fn [m] (merge auto-meta m)))))


(defn entity->datastore-entity [entity]
  (let [entity (auto-detect-meta entity)
        data (-> (::p/data entity)
                 (apply-to-db-meta-transforms (::p/meta entity))
                 transform-keywords-to-db-keywords)]
    (-> data
        (merge
         (entity->identity entity)
         (when-not (empty? (::p/meta entity))
           {:meta-anotations (meta->db-meta (::p/meta entity))}))
        (assoc :ds/indexes (data->indexes data)
               :meta-id (::p/id entity)))))


;;;; Db Entity -> Entity
;;;; * Transform keywords that where not annotated
;;;; * Transform data based on annotations

(defn datastore-entity->entity
  ([entity db-entity]
   (datastore-entity->entity api/unwrap-entity entity db-entity))
  ([unwrap entity db-entity]
   (let [db-entity (when (map? db-entity) (unwrap db-entity))
         entity-meta (::p/meta entity {})
         saved-meta (db-meta->meta (:meta-anotations db-entity []))
         meta (merge entity-meta saved-meta)
         data (when (map? db-entity)
                (-> db-entity
                    transform-db-keywords-to-keywords
                    (apply-from-db-meta-tranforms meta)
                    (dissoc :meta-anotations :meta-id)))]
     (assoc entity
            ::p/data data
            ::p/meta meta))))


;;;; Query -> GQL

(defn query->gql [query]
  (let [base-gql (str "select * from `" (::p/entity-key query) "`")
        params (::p/params query)]
    (if (empty? params)
      base-gql
      (str
       base-gql
       " where "
       (->> params
            (map (fn [[k v]]
                   (let [v (if (string? v) (str "\"" v "\"") v)]
                     (str "`" (c/keyword->ns-str k) "` = " v))))
            (string/join " and "))))))


(defn children-query->qgl [query]
  (str "select * "
       (if-let [kind (::p/entity-key query)]
         (str"from `" kind "`")
         "")
       " where __key__ has ancestor Key("
       (::p/parent-key query) ","
       "'" (::p/parent-id query) "'"
       ")"))


;;;; DB


(defn- query-db-entity
  "unwrap allows you to hook in before and after entity unwrapping (fn [db-entity unwrapped-entity])"
  [->gql client wrap-query query & {:keys [unwrap]
                                    :or {unwrap (fn [_ e] e)}}]
  (let [->domain-entity (fn [de] (api/domain-entity de :keyfn (::p/keyfn query)))
        gql (->gql query)
        query-params (wrap-query (entity->namespace query))
        results (let [l (::p/limit query)
                      o (::p/offset query)
                      p (->> [(when l [:limit l])
                              (when o [:offset o])]
                             (remove nil?)
                             (reduce into))]
                  (apply api/run-query client gql query-params p))
        db-entity->entity (fn [db-e]
                            (unwrap db-e (datastore-entity->entity ->domain-entity query db-e)))]
    (->> results
         (map (fn [db-e]
                (let [key-info (api/db-entity->key-info db-e)]
                  (-> db-e
                      db-entity->entity
                      (assoc ::p/id (:ds/id key-info)
                             ::p/key-info key-info)
                      (dissoc ::p/params ::p/keyfn))))))))


(defn db
  [{:keys [client project-id] :as options}]
  (let [client (api/api-client options)
        wrap-entity (fn [db-ns db-entity] ((api/wrap-entity project-id db-ns) db-entity))
        wrap-query (fn [db-ns] ((api/wrap-query-params project-id db-ns) api/weak-read-option))]

    (reify
      Db

      (upsert-entity [_ entity]
        (api/upsert-entity
         client
         (wrap-entity (entity->namespace entity) (entity->datastore-entity entity)))
        nil)


      (upsert-entities [_ entities]
        (api/upsert-entities
         client
         (->> entities
              (map #(wrap-entity (entity->namespace %) (entity->datastore-entity %)))))
        nil)

      (lookup-entity [_ entity]
        (let [db-entity
              (api/lookup-entity
               client
               (wrap-entity (entity->namespace entity) (entity->identity entity)))]
          (datastore-entity->entity entity db-entity)))

      (query-entity [_ query]
        (query-db-entity query->gql client wrap-query query))

      (delete-entity [_ entity]
        (api/delete-entity
         client
         (wrap-entity (entity->namespace entity) (entity->datastore-entity entity)))
        nil)

      (query-children [_ query]
        (query-db-entity children-query->qgl client wrap-query query
                         :unwrap (fn [db-e e]
                                   (if (::p/entity-key query)
                                     e
                                     (let [kind
                                           (-> db-e :entity :key :path last :kind (or ""))]
                                       (assoc e ::p/entity-key kind))))))

      (query-gql [_ gql-query]
        (query-db-entity ::p/gql client wrap-query gql-query)))))


(defmethod ig/init-key :simply.gcp.persistence.db/db
  [_ {:keys [client project-id] :as options}]
  (db options))

(comment

  (defn- db_ []
    (db {:project-id "dogmatix"
         :client (simply.gcp.auth/emulator-client :datastore-host "localhost" :datastore-port 8080)}))

  (def e_
    (p/entity
     :db-namespace (p.ns/db-namespace "TESTDB")
     :entity-key "test"
     :id "barone1114"

     :data {:a 1
            :b "sdfkhjjkhdf"
            :m {:foo :bar
                :foo-m :bar}
            :v [{:foo :bar}]
            :kw :foo
            :kwns :foo/bar
            :kw-m :foo
            :kwns-m :foo/bar
            :timestamp 1111111111
            :timestamp-m 1111111111
            :some-set #{1 2 3}


            :payment-status :collect
            :payment-status-as-meta :dont-collect

            }

     :meta {:kw-m :keyword
            :kwns-m :keyword
            [:m :foo-m] :keyword
            :timestamp-m :timestamp
            :payment-status-as-meta :keyword}))


  (protocol/upsert-entity
   (db_)
   e_)

  (protocol/lookup-entity
   (db_)
   (dissoc e_ ::p/data ::p/meta))


  (= e_ (protocol/lookup-entity
         (db_)
         (dissoc e_ ::p/data)))


  (protocol/delete-entity
   (db_)
   (assoc e_ ::p/data {}))

  [
   (protocol/query-entity
    (db_)
    (p/query :db-namespace p.ns/Contracts
             :entity-key "contract-search-fields"
             :limit 2
             :offset 3))



   (protocol/query-children
    (db_)
    (p/children-query :db-namespace p.ns/Sales
                      :entity-key "answer"
                      :parent-key "interaction"
                      :parent-id "002e91a2-537d-49de-9049-49d06e9374a9"
                      ))
   ]

  ;;; All namespaces
  (protocol/query-gql
   (db_)
   (p/gql-query :gql "select __key__ from __namespace__"))


  (protocol/query-gql
   (db_)
   (p/gql-query
    :db-namespace p.ns/Common
    :gql "select __key__ from __kind__"))


  (protocol/upsert-entities
   (db_)
   [(p/entity :db-namespace (p.ns/db-namespace "TEST")
              :entity-key "un"
              :id "one"
              :data {:one "1"})
    (p/entity :db-namespace (p.ns/db-namespace "TEST")
              :entity-key "du"
              :id "two"
              :data {:two "2"})])

  )
