(ns simply.wording-config-exporter
  (:require [simply.wording-config-parser :as config-parser]
            [clojure.data.csv :as csv]
            [clojure.java.io :as io]))


(defn- progress
  [& messages]
  (prn (apply str (interpose " " messages))))

(defn- all-the-data
  [partners config-directory config-types]
  (prn partners)
  (prn config-directory)
  (prn config-types)
  (into {}
        (map #(assoc {} % (config-parser/parse-config
                           :types config-types
                           :partner %
                           :config-directory config-directory))
             partners)))


(defn- all-the-keys
  [data config-type]
  (into #{}
        (filter #(not (nil? %))
                (flatten
                 (reduce (fn [result entry]
                           (let [value (get (second entry) config-type)]
                             (conj result (if (map? value)
                                            (keys value)
                                            nil))))
                         [] data)))))


(defn- cell
  [data partner config-type config-key]
  (progress "Getting cell data for" partner config-type config-key)
  (get-in data [partner config-type config-key]))

(defn- rows
  [partners config-directory config-types config-type]
  (let [data (all-the-data partners config-directory config-types)]
    (progress "Getting rows for" config-type)
    (cons (cons "keys" partners)
          (map (fn [config-key]
                 (cons
                  config-key
                  (reduce #(conj %1 (cell data %2 config-type config-key)) [] partners)))
               (all-the-keys data config-type)))))


(defn- write-to-csv
  [output-dir config-type data]
  (let [path (str output-dir "export-config-" (name config-type) ".csv")]
    (progress "Writing" path)
    (with-open [file (io/writer path)]
      (csv/write-csv file data))))


(defn export-data
  [partners config-directory config-types]
  (map (fn [config-type]
         (progress "Exporting" config-type)
         (rows partners config-directory config-types config-type))
       config-types))

(defn export-to-csv
  [partners config-directory config-types output-dir]
  (map (fn [config-type]
         (progress "Exporting" config-type)
         (write-to-csv output-dir config-type (rows partners config-directory config-types config-type)))
       config-types))


(comment

  (def config-directory "wording-config")

  (def partners ["base" "simply" "old-mutual" "absa"])

  ;; you can make this absolutely absolute - /Users/user/Documents/Simply/
  (def output-directory "resources/")

  (def config-types [:email-signatures
                     :parameterised-wording
                     :test-emails
                     :test-wording])

  ;; Only get the data for the given config types for each partner in a specific directory
  (export-data partners config-directory config-types)

  ;; Export that data to csv files in a specific directory
  (export-to-csv partners config-directory config-types output-directory)

)
