(ns simply.deps
  (:require [integrant.core :as ig]
            [simply.errors :as e]
            [simply.cqrs :as cqrs]))

(defonce ^:private *dependencies (atom {}))


(defn get-dep
  [k & {:keys [satisfies]}]
  (let [dep (get @*dependencies k)]

    (when (not (some? dep))
      (e/throw-app-error
       "Dependency not defined, ensure integrant is started and the simply.deps/deps key is added to your config and has the :dep key set"
       {:dep k}))

    (when (and satisfies (not (satisfies? satisfies dep)))
      (e/throw-app-error
       "Dependency does not satisfy expected protocol"
       {:dep k
        :expects (str (:on satisfies))}))

    dep))


(defn- cqrs-system []
  (get-dep :cqrs-system :satisfies cqrs/CqrsSystem))


(defn send-messages [topic coll]
  (cqrs/send-messages (cqrs-system) topic coll))


(defn send-message [topic message]
  (send-messages topic [message]))


(defn handle-command [type data user]
  (let [handle (cqrs/get-command-handler (cqrs-system))]
    (handle type data user)))


(defn require-actions [actions]
  (cqrs/request-actions (cqrs-system) actions))


(defn set-deps! [deps-map]
  (reset! *dependencies deps-map))


(defmethod ig/init-key :simply.deps/deps [_ dependencies]
  (set-deps! dependencies))


(defmethod ig/halt-key! :simply.deps/deps [_ _]
  (set-deps! {}))
