(ns simply.scheduling.core
  (:require [taoensso.timbre :as log]
            [clojure.spec.alpha :as s]
            [overtone.at-at :as at]))


(defprotocol  ^:deprecated ScheduleDb
  (should-run-interval-job? [this job-key inteval]))


(defn second-interval ^:deprecated [s] (* s 1000))
(defn minute-interval ^:deprecated [m] (* m (second-interval 60)))
(defn hour-interval ^:deprecated [h] (* h (minute-interval 60)))


;;;; INTERVAL JOB

(defn interval? ^:deprecated [x] (and (int? x) (pos? x)))

(s/def ::job-key string?)
(s/def ::interval interval?)
(s/def ::check-interval interval?)
(s/def ::job-f fn?)
(s/def ::interval-job (s/keys :req-un [::job-key
                                       ::interval
                                       ::check-interval
                                       ::job-f]))


(defn- interval-job? [job] (s/valid? ::interval-job job))

(defn ^:deprecated interval-job
  "Constructs an interval-job to be used with `start-interval-job`
  * Params:
      job-key: a unique string identifying the job
      interval: the interval period in milliseconds
      job-f: the function to execute when running a job"
  [job-key, interval, job-f]
  {:post [(interval-job? %)]}
  {:job-key job-key
   :interval interval
   :check-interval (int (/ interval 4))
   :job-f job-f})


(defn ^:deprecated with-check-interval
  "Change the check-interval frequency at which the schedule db is checked if a job should run"
  [check-interval interval-job]
  {:post [(interval-job? %)]}
  (assoc interval-job :check-interval check-interval))


(defn- interval-job-runner
  [schedule-db {:keys [job-key interval job-f] :as job}]
  {:pre [(satisfies? ScheduleDb schedule-db) (interval-job? job)]}
  (fn []
    (let [should-run? (should-run-interval-job? schedule-db job-key interval)]
      (log/info (if should-run?
                  (format "Running Job: %s" job-key)
                  (format "Job Already Ran: %s" job-key)))
      (when should-run?
        (try
          (job-f)
          (catch Exception e
            (log/error e (format "Could not complete job for %s" job-key))))))))


(defn ^:deprecated start-interval-job
  "Start an interval job.

   An interval job is built to be scheduled on many services simultaneously, but only run on one instance at a time. This allows stateless services to define and run jobs without the need to have a single job running service. All server instances will check a `ScheduleDb` (that is shared by all services) to determine if a job is allowed to run.
  The scheduler will check the `ScheduleDb` based on the job check interval. If the interval time has elapsed (as determined by the `ScheduleDb`) the interval job will run.

  There is a 10 second waiting period before the scheduler starts

  Example:
    interval = 6 hours
    check interval = 1 hour
    Job will run every 6-7 hours

  Returns a `stop-interval-job` function"
  [schedule-db interval-job]
  {:pre [(satisfies? ScheduleDb schedule-db) (interval-job? interval-job)]}
  (let [initial-delay 10000
        pool (at/mk-pool)
        stop-job #(at/stop-and-reset-pool! pool :strategy :kill)
        run-job (interval-job-runner schedule-db interval-job)
        check-interval (:check-interval interval-job)]
    (at/interspaced check-interval run-job pool :initial-delay initial-delay)
    stop-job))
