(ns simply.topic-watcher
  (:require [org.httpkit.server :as sockets]
            [compojure.core :refer [GET routes]]
            [integrant.core :as ig]
            [clojure.pprint :refer [pprint]]
            [cheshire.core :as jsn]
            [clojure.string :as string]
            [hiccup.core :as hiccup]))

(defonce ^:private *server (atom nil))
(defonce ^:private *channels (atom #{}))

(def ^:private topic-modifiers
  {"sms" {:f [:type :number :reference]
          :color "#4caf50"}
   "email" {:f [:type :to]
            :color "#2196f3"}
   "events" {:f [:type]
            :color "#9c27b0"}
   "slack" {:f [:type]
            :color "#f44336"}
   "collections" {:color "#ff9800"}
   "companies" {:color "#ffc107"}
   "contracts" {:color "#8bc34a"}
   "details-capture" {:color "#ffc107"}
   "leads" {:color "#3f51b5"}
   "new-contracts" {:color "#e91e63"}
   "simply-cron-jobs" {:color "#795548"}}
  )


(defn- now-str [] (.format (java.text.SimpleDateFormat. "HH:mm") (java.util.Date.)))


(defn- apply-modifier [prefix topic data]
  (let [m (if (string/includes? topic "required-actions")
            {:color "#9e9e9e" :f [:type]}
            (get topic-modifiers topic {:color "black"}))]
    (cond-> {:name (str (now-str) " " prefix " -> " topic)
             :color (:color m)
             :data (with-out-str (clojure.pprint/pprint data))}

      (:f m)
      (update :name (fn [n]
                      (->> ((apply juxt (:f m)) data)
                           (map str)
                           (string/join "  |  ")
                           (str n ":  ")))))))


(defn- can-notify? []
  (true? (:started? @*server)))


(defn- notify-clients [msg]
  (when (can-notify?)
    (doseq [channel @*channels]
      (sockets/send! channel (jsn/generate-string msg)))))


(defn message-sent [topic data]
  (when (can-notify?)
    (notify-clients (apply-modifier "sent" topic data))))


(defn messages-sent [topic coll]
  (when (can-notify?)
    (doseq [data coll]
      (message-sent topic data))))


(defn message-received [subscription data]
  (when (can-notify?)
    (notify-clients (apply-modifier "received" subscription data))))


(defn messages-received [subscription coll]
  (when (can-notify?)
    (doseq [data coll]
      (message-received subscription data))))


(defn- connect! [channel]
  (swap! *channels conj channel))


(defn- disconnect! [channel status]
  (swap! *channels #(remove #{channel} %)))


(defn- web-socket [request]
  (sockets/with-channel request cnl
    (connect! cnl)
    (sockets/on-close cnl (partial disconnect! cnl))))


(defn- start-server [port]
  (when-not (:started? @*server)
    (reset! *server {:started? true
                     :port port
                     :stop (sockets/run-server web-socket {:port port})})))


(defn- stop-server []
  (let [{:keys [started? stop]} @*server]
    (when started?
      (stop))
    (reset! *server nil)))


(defn- html [port]
  (str
   "<!DOCTYPE html>"
   (hiccup/html
    [:html
     [:head
      [:script {:type "text/javascript"}
       (str "var endpoint = 'ws://localhost:" port "';")
       "var socket = new WebSocket(endpoint);"
       "socket.onmessage =
          function(event) {
            var data = JSON.parse(event.data);
            var e = document.createElement('details');
            var s = document.createElement('summary');
            var d = document.createElement('pre');
            d.innerText = data.data;
            s.style.color = data.color;
            s.innerText = data.name;
            e.appendChild(s);
            e.appendChild(d);
            document.getElementById('app').prepend(e);
          };"]]
     [:body
      [:span "pubsub items will be displayed newest to oldest..."]
      [:div#app]]])))


(defn handler
  "Put this in your compojure handler"
  []
  (routes
   (GET "/repl/pubsub" request
        (let [{:keys [started? port]} @*server]
          (if (true? started?)
            {:status 200 :body (html port) :headers {}}
            {:status 404})))))


(defmethod ig/init-key :simply.topic-watcher/server
  [_ {:keys [port run?]
      :or {port 3999
           run? false}}]
  (when (true? run?)
    (start-server port)))


(defmethod ig/halt-key! :simply.topic-watcher/server
  [_ _]
  (stop-server))


(comment
  (start-server 3997)
  (message-sent "email" {:to ["0980980988"] :type "test-type" :reference "abcde"})
  (stop-server))
