(ns simply.gcp.errorreporting
  (:require [cheshire.core :as json]
            [org.httpkit.client :as http]
            [simply.gcp.util :as util]
            [integrant.core :as ig]))


(defn- event-data [{:keys [?err msg_ instant ?ns-str ?line]} context]
  (let [msg @msg_
        e (if ?err (Exception. msg ?err) (Exception. msg))
        stack-writer (java.io.StringWriter.)]
    (.printStackTrace e (java.io.PrintWriter. stack-writer))
    {:message (str stack-writer)
     :eventTime (util/unparse-zulu-date-format instant)
     :serviceContext context
     :context {:reportLocation {:functionName (str ?ns-str)
                                :lineNumber (or ?line 0)}}}))


(defn- appender-fn
  [{:keys [project-id api-key]} context]
  (let [url (format
             "https://clouderrorreporting.googleapis.com/v1beta1/projects/%s/events:report?key=%s"
             project-id api-key)]
    (fn [timbre-data]
      (try
        (let [event (event-data timbre-data context)
              request
              {:method :post
               :url url
               :body (json/generate-string event)}]
          (http/request request))
        (catch Exception e)))))


(defn- get-context [{:keys [container-name container-id]}]
  {:service container-name
   :version container-id
   :resourceType "container"})


(defn appender
  "timbre appender for GCP Error Reporting"
  [options]
  {:enabled?   true
   :async?     true
   :min-level  :error
   :rate-limit nil
   :output-fn  :inherit
   :fn (appender-fn options (get-context options))})


(defmethod ig/init-key :simply.gcp.errorreporting/appender
  [_ {:keys [config] :as options}]
  (let [{:keys [project-id api-key container-name container-id]} config]
    {:name :clouderrors
     :appender (appender config)}))
