(ns simply.gcp.pubsub.core
  (:require [cheshire.core :as json]
            [taoensso.timbre :as log]
            [simply.errors :as e]
            [overtone.at-at :as at]
            [clojure.spec.alpha :as s]
            [integrant.core :as ig]))


;;;; API

(defn- pubsub-message [encode data]
  {:attributes
   (merge {}
          (when-let [request-id (get data :requestId)]
            {:requestId request-id}))
   :data (encode data)})


(defn- publish-messages [encoder {:keys [topic messages]}]
  {:method :post
   :url (format "topics/%s:publish" topic)
   :body {:messages (map #(pubsub-message encoder %) messages)}})


(defn- ack
  [{:keys [ack-id subscription]}]
  {:method :post
   :url (format "subscriptions/%s:acknowledge" subscription)
   :body {:ackIds [ack-id]}})


(defn- nack
  [{:keys [ack-id subscription ack-deadline-seconds]}]
  {:method :post
   :url (format "subscriptions/%s:modifyAckDeadline" subscription)
   :body {:ackIds [ack-id]
          :ackDeadlineSeconds ack-deadline-seconds}})


(defn- pull
  [{:keys [subscription]}]
  {:method :post
   :url (format "subscriptions/%s:pull" subscription)
   :body {:returnImmediately true
          :maxMessages 1}})


(defn- pull-message
  [subscription client]
  (let [*ack-id (atom nil)
        sub-key (:key subscription)
        decoder (:decoder subscription)
        f (:handler subscription)]
    (try
      (let [response (client (pull {:subscription sub-key}))
            body (:body response)]
        (if body
          (let [msg (-> body (json/parse-string keyword) :receivedMessages first)]
            (when msg
              (let [ack-id (:ackId msg)
                    data (-> msg
                             (get-in [:message :data])
                             decoder)]
                (do
                  (reset! *ack-id ack-id)
                  (f data)
                  (client (ack {:ack-id ack-id :subscription sub-key}))))))
          (log/infof "Subscription Message no body. Status: %s | Method: %s | Url: %s"
                     (:status response) (get-in response [:opts :method]) (get-in response [:opts :url]))))
      (catch Exception e
        (log/error e (format "Could not process message for subscription %s | Cause: %s"
                             sub-key (:cause (Throwable->map e))))
        (when-let [ack-deadline-seconds (:ack-deadline-seconds subscription)]
          (when-let [ack-id @*ack-id]
            (client (nack {:ack-id ack-id
                           :subscription sub-key
                           :ack-deadline-seconds ack-deadline-seconds}))))))))


(defn- create-topic [{:keys [topic]}]
  {:method :put
   :url (format "topics/%s" topic)})


(defn- create-subscription [{:keys [topic subscription opts]}]
  {:method :put
   :url (format "subscriptions/%s" subscription)
   :body (merge opts {:topic (format "topics/%s" topic)})})


(defn- wrap-pubsub-params [root project-id client]
  (fn [config]
    (let [body
          (when-let [body (:body config)]
            {:body
             (json/generate-string
              (if-let [topic (:topic body)]
                (assoc body :topic (format "projects/%s/%s" project-id topic))
                body))})]
      (client
       (->
        (merge
         config
         {:url (format "%sprojects/%s/%s" root project-id (:url config))}
         {:scope "https://www.googleapis.com/auth/pubsub"}
         body))))))


(defn- already-exists? [{:keys [status body]}]
  (if (= 409 status)
    (let [err (-> body (json/parse-string keyword) :error)
          err-message (:message err)
          err-status (:status err)]
      (or (= err-message "Topic already exists")
          (= err-message "Subscription already exists")
          (= err-status "ALREADY_EXISTS")))
    false))


(defn- extract-error-message [error]
  (str "Pubsub Request Failed: "
       (cond
         (instance? Throwable error) (:cause (Throwable->map error))
         (string? error)             error
         :else                       "")))


(defn- wrap-error-handling [client]
  (fn [config]
    (let [{:keys [status body error opts] :as response} @(client config)]
      (when (or (nil? status) (<= 300 status) (> 200 status))
        (when-not (already-exists? response)
          (e/throw-app-error (extract-error-message error) {:status status
                                                            :body   body
                                                            :error  error
                                                            :url    (:url opts)
                                                            :method (:method opts)})))
      response)))

(defn- wrap-client [root project-id client]
  (->> client
       (wrap-pubsub-params root project-id)
       wrap-error-handling))


(defn- ensure-topic-exists [topic client]
  (let [{:keys [status body error opts] :as response}
        (client (create-topic {:topic (:key topic)}))]
    (when-not (contains? #{200 409} status)
      (e/throw-app-error
       (str "Could Not Create Topic " (:key topic) {:url (:url opts)
                                                    :status status
                                                    :body body
                                                    :error error})))
    topic))


(defn- ensure-subscription-exists [subscription client]
  (let [{:keys [status body error opts] :as response}
        (client (create-subscription {:topic (:topic subscription)
                                      :subscription (:key subscription)
                                      :opts (:opts subscription)}))]
    (when-not (contains? #{200 409} status)
      (e/throw-app-error
       (str "Could Not Create Subscription " (:key subscription) {:url (:url opts)
                                                                  :form-params (:form-params opts)
                                                                  :status status
                                                                  :body body
                                                                  :error error})))
    subscription))


(def pull-interval-ms 10)
(def pull-initial-delay 2000)


(defn- subscribe
  [{:keys [pull-interval] :as subscription} client]
  (let [pool (at/mk-pool)
        unsubscribe #(at/stop-and-reset-pool! pool :strategy :kill)]
    (at/every pull-interval #(pull-message subscription client) pool :initial-delay pull-initial-delay)
    unsubscribe))


(s/def :pubsub/key string?)
(s/def :pubsub/pull-interval pos?)
(s/def :pubsub/decoder fn?)
(s/def :pubsub/encoder fn?)
(s/def :pubsub/handler fn?)
(s/def :pubsub/topic (s/keys :req-un [:pubsub/key
                                      :pubsub/encoder]))
(s/def :pubsub/topics (s/coll-of :pubsub/topic))
(s/def :pubsub-sub/topic string?)
(s/def :pubsub/ack-deadline-seconds (fn [v] (and (>= v 10) (<= v 600))))
(s/def :pubsub/subscription (s/keys :req-un [:pubsub/key
                                             :pubsub-sub/topic
                                             :pubsub/pull-interval
                                             :pubsub/decoder
                                             :pubsub/handler]
                                    :opt-un [:pubsub/ack-deadline-seconds]))
(s/def :pubsub/subscriptions (s/coll-of :pubsub/subscription))


(defn topic [key encoder]
  {:post [(s/valid? :pubsub/topic %)]}
  {:key key :encoder encoder})


(defn subscription
  ([key topic decoder handler] (subscription key topic decoder handler pull-interval-ms))
  ([key topic decoder handler pull-interval]
   {:post [(s/valid? :pubsub/subscription %)]}
   {:key key :topic topic :decoder decoder :pull-interval pull-interval :handler handler}))


(defn with-ack-deadline [subscription ack-deadline-seconds]
  (assoc-in subscription [:opts :ackDeadlineSeconds] ack-deadline-seconds))


(defn with-opts
  "Adds extra options to the subscription.
  See https://cloud.google.com/pubsub/docs/reference/rest/v1/projects.subscriptions/create
  for a complete list."
  [subscription opts]
  (assoc subscription :opts opts))


(defprotocol Pubsub
  (init [this])
  (subscribe-all [this])
  (unsubscribe-all [this])
  (publish [this topic-key messages]))


(defn pubsub
  [client topics subscriptions]
  {:pre [(s/valid? :pubsub/topics topics) (s/valid? :pubsub/subscriptions subscriptions)]}
  (let [*subs (atom {})
        topics-map (->> topics
                        (map (juxt :key identity))
                        (into {}))]
    (reify
      Pubsub

      (init [this]
        (let [topic-keys (set (map :key topics))
              sub-topics (->> subscriptions
                              (map :topic)
                              (filter #(not (contains? topic-keys %)))
                              (map #(topic % identity)))]
          (doseq [t topics]
            (ensure-topic-exists t client))
          (doseq [t sub-topics]
            (ensure-topic-exists t client))
          (doseq [s subscriptions]
            (ensure-subscription-exists s client))
          this))

      (subscribe-all [this]
        (doseq [s subscriptions]
          (swap! *subs #(assoc % (:key s) (subscribe s client))))
        this)

      (unsubscribe-all [this]
        (doseq [[k unsubscribe] @*subs]
          (unsubscribe)
          (swap! *subs #(dissoc % k)))
        this)

      (publish [this topic-key messages]
        (if-let [t (get topics-map topic-key)]
          (client
           (publish-messages (:encoder t) {:topic topic-key
                                           :messages messages}))
          (e/throw-app-error (str topic-key " is not a valid topic")))))))


;;;; CLIENTS

(defn production-client
  "Wraps a simply.gcp.auth client to run against pubsub in production"
  [project-id auth-client]
  (wrap-client "https://pubsub.googleapis.com/v1/" project-id auth-client))


(defn emulator-client
  "Wraps a simply.gcp.auth client to run against pubsub emulator"
  [port project-id auth-client]
  (wrap-client (str "http://localhost:" port "/v1/") project-id auth-client))


(defmethod ig/init-key :simply.gcp.pubsub.core/production-client
  [_ {:keys [client project-id]}]
  (production-client project-id client))


(defmethod ig/init-key :simply.gcp.pubsub.core/emulator-client
  [_ {:keys [client project-id port]}]
  (emulator-client port project-id client))
