(ns simply.gcp.url
  (:require [clojure.string :as string]
            [clojure.data.json :as json]
            [org.httpkit.client :as client]
            [simply.core :refer [ShortenUrl] :as s.core]
            [integrant.core :as ig]
            [taoensso.timbre :as log]
            [cemerick.url :as url]))


;;;; URLS must contain `https://{optionally.anything.}simply.co.za{optionally anything}`
;;;; Use the following whitelisted regex: `https:\/\/?(.*)simply+\.co\.za?(.*)`


(defn- shorten-request [key long-url]
  {:method :post
   :url (format "https://firebasedynamiclinks.googleapis.com/v1/shortLinks?key=%s" key)
   :body (json/write-str {:longDynamicLink (str "https://smply.page.link?link=" (url/url-encode long-url))
                          :suffix {:option "SHORT"}})
   :headers {"Content-Type" "application/json"}})


(defn- shorten [key long-url]
  (let [request (shorten-request key long-url)
        response @(client/request request)
        valid? (= 200 (:status response))]
    (when-not valid?
      (log/error (str "Could Not Shorten: long-url => " (:body response))))
    (if valid?
      (-> response :body json/read-str (get "shortLink"))
      long-url)))


(defn url-shortner [api-key]
  (reify
    ShortenUrl
    (try-shorten [this long-url]
      (if api-key
        (shorten api-key long-url)
        long-url))))


(defmethod ig/init-key :simply.gcp.url/url-shortner
  [_ {:keys [api-key]}]
  (url-shortner api-key))


(comment
  ;;;; BAD KEY returns LONG URL and LOGS Error
  (s.core/try-shorten (url-shortner "123") "http://sneakycode.net")

  ;;;; GOOD KEY will NOT shorten non simply
  (let [api-key ""]
    (s.core/try-shorten (url-shortner api-key) "http://sneakycode.net"))

  ;;;; GOOD KEY and simply Link will shorten
  (let [api-key ""]
    (s.core/try-shorten (url-shortner api-key) "https://simply.co.za"))


   ;;;; GOOD KEY and url params should work
  (let [api-key ""]
    (s.core/try-shorten (url-shortner api-key) "https://simply.co.za?foo=1&bar=1"))
  )
