(ns simply.wording-config
  (:require [clojure.string :as string]
            [cljstache.core :as mustache-parser]
            [simply.wording-config.templating :as wc.templating]))

(def default-type :all)


(def wording-config (atom {}))

(defn current-wording []
  @wording-config)


(defn load-config! [& {:keys [config]}]
  (reset! wording-config config))


(defn throw-if-config-nil [value type key]
  (if (nil? value)
    (let [error-message (str "Cannot find wording configuration for type = " type
                             ", key = " key ". Please check your wording configuration files.")]
      (throw (Exception. error-message)))))


(defn get-configured-value [type key]
  (get-in @wording-config
          [type key]))


(defn has-key? [& {:keys [type key]
                   :or {type default-type}}]

  (not (nil?
        (get-configured-value type
                              key))))


(defn get-wording [& {:keys [type
                             key
                             apply-template-params?
                             params
                             wording-keys-as-params]
                      :or {type default-type
                           apply-template-params? true
                           params {}}}]

  (let [value      (get-configured-value type key)

        ;; Note: All current wording is merged into the template params
        ;; making any configured wording available to all templates.
        all-params (merge {}
                          (current-wording)
                          params)]

    (throw-if-config-nil value type key)

    (if apply-template-params?
      (if (coll? value)
        (map #(mustache-parser/render % all-params)
             value)
        (mustache-parser/render value
                                all-params))
      ;; Raw template without params applied
      value)))


(defn render "Warps cljstache render"
  [template-str params & {:keys [apply-template-params?] :or {apply-template-params? false}}]
  (let [params (if apply-template-params?
                 (merge {} (current-wording) params)
                 params)]
    (mustache-parser/render template-str params)))


(defn apply-templating "Applies #sw/tmpl templating on a clojure data strucutre"
  [template params & {:keys [apply-template-params?] :or {apply-template-params? false}}]
  (let [params (if apply-template-params?
                 (merge {} (current-wording) params)
                 params)]
    (wc.templating/apply-template template params)))


(defn get-email [& {:keys [type
                           key
                           signature
                           apply-template-params?
                           params
                           wording-keys-as-params]
                    :or {type default-type
                         apply-template-params? true
                         params {}}}]

  (let [email (get-wording :type type
                           :key key
                           :apply-template-params? apply-template-params?
                           :params params
                           :wording-keys-as-params wording-keys-as-params)]

    (if (nil? signature)
      email
      ;; Include signature
      (let [signature           (get-wording
                                 :type (or (:type signature)
                                           type)
                                 :key (:key signature)
                                 :apply-template-params? apply-template-params?
                                 :params (or (:params signature)
                                             params)
                                 :wording-keys-as-params (:wording-keys-as-params signature))

            [subject body]      email

            body-with-signature (str body "\n\n"
                                     signature)]
        [subject body-with-signature]))))


(defn get-email-as-map [& options]
  (let [[subject body] (apply get-email options)]
    {:subject subject
     :body body}))


(defn get-value
  ([& ks]
   (get-in @wording-config ks)))
