(ns simply.gcp.auth
  (:require [org.httpkit.client :as http]
            [simply.errors :as e]
            [integrant.core :as ig]
            [clojure.string :as string])
  (:import com.google.api.client.googleapis.auth.oauth2.GoogleCredential
           java.io.FileInputStream
           java.util.Collections))


;;;; CREDENTIALS

(def *scoped-credentials (atom {}))


(defn reset-credentials! []
  (reset! *scoped-credentials {}))


(defn- scoped-token
  "returns an existing oauth token, or fetches one if expired"
  [credential scope]
  (when-not (contains? @*scoped-credentials scope)
    (swap! *scoped-credentials assoc scope credential))
  (let [cred (get @*scoped-credentials scope)]
    (cond-> cred
      (nil? cred) (e/throw-app-error "Google Credential Cannot be Nil" {:scope scope})
      (nil? (.getAccessToken cred)) .refreshToken
      (> 60 (.getExpiresInSeconds cred)) .refreshToken)
    (swap! *scoped-credentials assoc scope cred)
    (.getAccessToken cred)))


(defn- local-credential [scope keyfile-path]
  (-> (FileInputStream. keyfile-path)
      GoogleCredential/fromStream
      (.createScoped (Collections/singleton scope))))


;;;; CLIENTS

(defn- re-route-to-emulator-if-datastore
  "We do this here cause sometimes a hack is much much less work"
  [req]
  (if (string/starts-with? (:url req) "https://datastore.googleapis.com/v1/projects/")
    (let [uri (last (string/split (:url req) #":" 3))]
      (-> req
          (assoc :url (str "http://localhost:8080/v1/projects/dogmatix:" uri))))
    req))


(defn emulator-client
  "For use with datastore and pubsub emulators that run locally."
  [] (fn [req]
       (-> req
           re-route-to-emulator-if-datastore
           http/request)))


(defn- credentials-client
  "Used to access GCP api's while authenticating using a json keyfile."
  [cred {:keys [scope] :as request}]
  (http/request
   (assoc request
          :oauth-token (scoped-token cred scope))))


(defn keyfile-client [keyfile-path]
  (fn [{:keys [scope] :as request}]
    (credentials-client (local-credential scope keyfile-path) request)))


(defn default-credentials-client
  "Used by containers running on google cloud that have access to default credentials."
  []
  (fn [request]
    (credentials-client (GoogleCredential/getApplicationDefault) request)))


;;;; INTEGRANT

(defmethod ig/init-key :simply.gcp.auth/emulator-client [_ _] (emulator-client))


(defmethod ig/init-key :simply.gcp.auth/keyfile-client
  [_ {:keys [keyfile-path]}]
  (keyfile-client keyfile-path))


(defmethod ig/init-key :simply.gcp.auth/default-credentials-client [_ _]
  (default-credentials-client))
