(ns simply.persistence.core
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as string]
            [simply.errors :as e]
            [simply.deps :as deps]
            [simply.persistence.db :refer [Db] :as db]
            [integrant.core :as ig]))

;;;; ENTITIES

(def allowed-meta-annotations #{:keyword :timestamp})

(s/def ::string #(and (string? %) (not (string/blank? %))))
(s/def ::db-namespace (s/coll-of ::string :kind #(and (vector? %) (odd? (count %)))))
(s/def ::entity-key ::string)
(s/def ::parent-key ::entity-key)
(s/def ::id ::string)
(s/def ::parent-id ::id)
(s/def ::data map?)
(s/def ::meta (s/map-of (s/coll-of keyword? :kind vector? :min-count 1) allowed-meta-annotations))
(s/def ::entity (s/keys :req [::db-namespace
                              ::entity-key
                              ::id
                              ::data
                              ::meta]))


(defn- guard-against-invalid-entity [entity]
  (when-let [err (s/explain-data ::entity entity)]
    (e/throw-app-error "Persistence entity does not conform to spec"
                       (dissoc err :clojure.spec.alpha/value)))
  (when (> (count (::db-namespace entity)) 1)
    (e/throw-app-error "Ancestor keys are not implemented yet"
                       (dissoc entity ::data))))


(defn db-namespace [& paths]
  (vec paths))


(defn entity
  [& {:keys [db-namespace entity-key id data meta]
      :or {db-namespace []
           entity-key ""
           id ""
           data {}
           meta {}}}]
  {::db-namespace db-namespace
   ::entity-key entity-key
   ::id id
   ::data data
   ::meta (->> meta
               (map (fn [[k v]] [(if (keyword? k) [k] k) v]))
               (into {}))})


;;;; QUERIES

(s/def ::params (s/map-of keyword? (s/or :string string?
                                         :number number?
                                         :boolean boolean?)))
(s/def ::limit pos-int?)

(s/def ::offset (fn [n] (and (integer? n) (>= n 0))))

(s/def ::query (s/keys :req [::db-namespace
                             ::entity-key
                             ::params]
                       :opt [::limit ::offset]))

(s/def ::children-query
  (s/keys :req [::db-namespace
                ::parent-key
                ::parent-id]
          :opt [::limit
                ::offset
                ::entity-key]))

(defn query "a simple query that matches on and equality only"
  [& {:keys [db-namespace entity-key params limit offset]
      :or {db-namespace []
           entity-key ""
           params {}}}]
  (cond-> {::db-namespace db-namespace
           ::entity-key entity-key
           ::params params}
    (number? limit) (assoc ::limit limit)
    (number? offset) (assoc ::offset offset)))


(defn children-query
  "a simple query that returns children related to parent.
   omit entity key for search across all entities"
  [& {:keys [db-namespace entity-key parent-key parent-id limit offset]
      :or {db-namespace []
           parent-key ""
           params {}}}]
  (cond-> {::db-namespace db-namespace
           ::parent-key parent-key
           ::parent-id parent-id}
    (number? limit) (assoc ::limit limit)
    (number? offset) (assoc ::offset offset)
    entity-key      (assoc ::entity-key entity-key)))


(defn- guard-against-invalid-query [query]
  (when-let [err (s/explain-data ::query query)]
    (e/throw-app-error "Persistence query does not conform to spec" err)))


(defn- guard-against-invalid-children-query [query]
  (when-let [err (s/explain-data ::children-query query)]
    (e/throw-app-error "Persistence children-query does not conform to spec" err)))


(s/def ::gql string?)


(s/def ::gql-query? (s/keys :req [::gql]
                            :opt [::limit ::offset]))


(defn- parse-gql-string [gql]
  (let [parts (reverse (clojure.string/split gql #"\s"))]
    (loop [all-parts parts
           parts parts
           limit nil
           offset nil]

      (if (or (empty? parts) (and limit offset))

        {:gql (clojure.string/join " " (reverse all-parts))
         :limit (when limit (Integer/parseInt limit))
         :offset (when offset (Integer/parseInt offset))}

        (let [[n t] (take 2 parts)
              t' (clojure.string/upper-case t)]
          (cond
            (= "LIMIT" t')
            (recur (drop 2 all-parts) (drop 2 parts) n offset)

            (= "OFFSET" t')
            (recur (drop 2 all-parts) (drop 2 parts) limit n)

            :else
            (recur all-parts [] limit offset)))))))


(defn gql-query
  "use for direct gql queries
  limit and offset can be passed in seperately or as part of the query string
  limit and offset as params take preference"
  [& {:keys [gql db-namespace limit offset]
          :or {gql ""
               db-namespace (db-namespace "")}}]
  (let [{limit' :limit offset' :offset gql' :gql} (parse-gql-string gql)
        limit (or limit limit')
        offset (or offset offset')]
    (cond-> {::gql gql'
             ::db-namespace db-namespace}
      (number? limit) (assoc ::limit limit)
      (number? offset) (assoc ::offset offset))))


(defn- guard-against-invalid-gql-query [gql-query]
  (when-let [err (s/explain-data ::gql-query gql-query)]
    (e/throw-app-error "Persistence gql-query does not conform to spec" err)))


;;;; API

(defn- ->db [] (deps/get-dep :simply.persistence.core/db :satisfies Db))


(defn upsert "inserts or updates the entity at id"
  [entity]
  (guard-against-invalid-entity entity)
  (db/upsert-entity (->db) entity))


(defn lookup "Finds a single entity by it's id"
  [entity]
  (guard-against-invalid-entity entity)
  (db/lookup-entity (->db) entity))


(defn find-all "finds all entities that match query"
  [query]
  (guard-against-invalid-query query)
  (db/query-entity (->db) query))


(defn find-all-children "finds all entities that that has parent"
  [children-query]
  (guard-against-invalid-children-query children-query)
  (db/query-children (->db) children-query))


(defn delete [entity]
  (guard-against-invalid-entity entity)
  (db/delete-entity (->db) entity))


(defn query-gql [gql-query]
  (guard-against-invalid-gql-query gql-query)
  (db/query-gql (->db) gql-query))



(comment

  (guard-against-invalid-entity
   (entity :db-namespace (db-namespace "SME")
           :entity-key "company-view"
           :id "Simply"
           :meta {[:foo] ::timestamp}))

  )


