(ns simply.gcp.datastore.core
  (:require [cheshire.core :as json]
            [simply.gcp.datastore.entities :as entities]
            [simply.errors :as e]
            [integrant.core :as ig]
            [camel-snake-kebab.extras :as camel-snake.extras]
            [camel-snake-kebab.core :as camel-snake]
            [clojure.string :as str]
            [simply.gcp.auth]))

;; entities and queries

(defn wrap-query-params
  "Returns a function that takes query parameters
   and adds project-id and namespace to the query-params."
  [project-id namespace]
  (fn [query-params]
    (merge query-params {:ds/namespace-id namespace
                  :ds/project-id project-id})))


(defn wrap-entity
  "Returns a function that takes an entity
   and adds project-id and namespace on the entity."
  [project-id namespace]
  (fn [entity]
    (merge entity {:ds/namespace-id namespace
                   :ds/project-id project-id})))


(defn unwrap-entity
  "Removes datastore keys from the entity"
  [entity]
  (dissoc entity :ds/kind :ds/indexes :ds/ancestor :ds/namespace-id :ds/name :ds/id))


(defn- with-clojure-keys [m]
  (camel-snake.extras/transform-keys camel-snake/->kebab-case-keyword m))


(defn domain-entity
  "turns a datastore entity into a domain entity removing datastore keys"
  [m]
  (-> m entities/parse-entity unwrap-entity with-clojure-keys))


(defn unwrap-entity-with-id
  "Removes all datastore keys except id from the entity"
  [entity]
  (-> entity
      (dissoc :ds/kind :ds/indexes :ds/ancestor :ds/namespace-id)
      (assoc :id (or (:ds/id entity)
                                                    (:ds/name entity)))))


(defn domain-entity-with-id
  "turns a datastore entity into a domain entity removing all datastore keys except id"
  [m]
  (-> m entities/parse-entity unwrap-entity-with-id with-clojure-keys))


(defn db-entity->id [m]
  (let [path (get-in m [:entity :key :path])]
    (:name (last path))))


(defn name-key [kind name]
  {:ds/kind kind :ds/name name})


(defn db-uuid [v] (if (uuid? v)
                    (.toString v)
                    v))


(defn app-uuid [v] (if (string? v)
                     (try (java.util.UUID/fromString v)
                          (catch IllegalArgumentException _ v))
                     v))


(defn db-date [v] (if (number? v)
                    (java.util.Date. v)
                    v))


(defn app-date [v] (try (.getTime v)
                        (catch IllegalArgumentException _ v)))


(def entity-help
  "* Use wrap-entity to add project-id and namespace
   * Entity needs the following keys
      - :ds/kind  String
      - either :ds/name  String    or    :ds/id  Int
      - :ds/indexes #{<keys of fields that needs to be indexed>}")

;; api


(def start-transaction-request-params
  {:method :post
   :url ":beginTransaction"})


(defn- decode [e] (json/parse-string e keyword))


(defn start-transaction [client]
  (-> start-transaction-request-params
      client
      :body
      decode
      :transaction))


;; Lookup

(defn- lookup-request [params]
  {:method :post
   :url ":lookup"
   :body
   {:readOptions
    (if-let [transaction (:transaction params)]
      {:transaction transaction}
      {:readConsistency (:readConsistency params "STRONG")})
    :keys (:keys params)}})


(defn- lookup-entity-request [entity]
  (let [key (entities/prepare-key entity)]
    (lookup-request
     (merge (select-keys entity [:transaction :readConsistency])
            {:keys [key]}))))


(defn lookup-entity
  "Fetches and Parses a datastore Entity. If not found returns nil.
   * Use the api-client fn to construct an api-client.
   * Use name-key to construct the entity key."
  [api-client entity-key]
  (let [response (api-client
                  (lookup-entity-request
                   entity-key))
        body (decode (:body response))]
    (when-let [entity (get-in body [:found 0])]
      (entities/parse-entity entity))))


;; Change

(def transactional-mode "TRANSACTIONAL")
(def non-transactional-mode "NON_TRANSACTIONAL")


(defn commit-request [params]
  {:method :post
   :url ":commit"
   :body
   (merge
    {:mode (:mode params)
     :mutations (:mutations params)}
    (when-let [transaction (:transaction params)]
      {:transaction transaction}))})


(defn- mutation
  ([t entity] (mutation identity t entity))
  ([transform t entity] {t (transform (entities/prepare-entity entity))}))

(defn- entities-request [mutation-fn params entity-coll]
  (commit-request
   (merge
    params
    {:mutations (map mutation-fn entity-coll)})))


(defn upsert-mutation [entity] (mutation :upsert entity))
(defn- upsert-entities-request [params entity-coll]
  (entities-request upsert-mutation params entity-coll))
(defn- upsert-entity-request [entity]
  (upsert-entities-request {:mode non-transactional-mode} [entity]))


(defn upsert-entity
  "Creates or updates an entity.
   * Use the api-client fn to construct an api-client.
   * See entity-help for info regarding entities"
  [api-client entity]
  (-> entity
      upsert-entity-request
      api-client))


(defn upsert-entities
  "Creates or updates entities in a transaction.
   * Use the api-client fn to construct an api-client.
   * See entity-help for info regarding entities"
  [api-client entities]
  (let [transaction (start-transaction api-client)
        request (upsert-entities-request
                 {:mode transactional-mode
                  :transaction transaction}
                 entities)]
    (api-client request)))


(defn delete-mutation [entity] (mutation :key :delete entity))
(defn- delete-entities-request [params entity-coll]
  (entities-request delete-mutation params entity-coll))
(defn- delete-entity-request [entity]
  (delete-entities-request {:mode non-transactional-mode} [entity]))


(defn delete-entity
  "Creates or updates an entity.
   * Use the api-client fn to construct an api-client.
   * See entity-help for info regarding entities"
  [api-client entity]
  (-> entity
      delete-entity-request
      api-client))


(defn commit-mutations [api-client mutations]
  (let [transaction (start-transaction api-client)
        request (commit-request {:mode transactional-mode
                                 :transaction transaction
                                 :mutations mutations})]
    (api-client request)))


;; query

(def strong-read {:readConsistency "STRONG"})
(def weak-read {:readConsistency "EVENTUAL"})
(def strong-read-option {:read-option strong-read})
(def weak-read-option {:read-option weak-read})


(defn run-query-request [query params]
  {:method :post
   :url ":runQuery"
   :body
   {:partitionId (entities/prepare-partition params)
    :gqlQuery {:queryString query
               :allowLiterals true}}})


(defn run-query-next-page-request [query params cursor]
  {:method :post
   :url ":runQuery"
   :body
   {:partitionId (entities/prepare-partition params)
    :gqlQuery {:allowLiterals true
               :namedBindings {:cursor {:cursor cursor}}
               :queryString (str query " OFFSET @cursor")}}})


(def ^:private max-batch-size 300)

(defn run-query
  "Runs a qgl query
  * Use the api-client fn to construct an api-client.
  * Use wrap-query-params to add project-id and namespace to query-params
  * Pass in an optional limit and offset"
  [api-client gql-query query-params & {:keys [limit offset instrument?]
                                        :or {instrument? false}}]

  (when (.contains (.toUpperCase gql-query) "LIMIT")
    (e/throw-app-error "run-query should not have LIMIT spesified as part of the gql-query" {:query gql-query}))

  (when (.contains (.toUpperCase gql-query) "OFFSET")
    (e/throw-app-error "run-query should not have OFFSET spesified as part of the gql-query" {:query gql-query}))

  (when (number? limit)
    (when (> 1 limit)
      (e/throw-app-error "run-query limit should be > 0")))

  (when (number? offset)
    (when (> 0 offset)
      (e/throw-app-error "run-query offset should be >= 0")))

  (let [api-client' (if instrument? (fn [r] (prn "***") (prn r) (api-client r)) api-client)
        has-limit-set? (number? limit)
        offset (if (number? offset) (str " OFFSET " offset) "")
        batch-size (if has-limit-set? (min max-batch-size limit) max-batch-size)
        batched-gql-query (str gql-query " LIMIT " batch-size offset)]
    (loop [request (run-query-request batched-gql-query query-params)
           result []
           successive-empty-calls 0]

      (let [{:keys [batch query]} (-> request
                                      api-client'
                                      :body
                                      decode)
            {:keys [moreResults endCursor entityResults]} batch
            combined-results (concat result entityResults)
            more-after-limit-result? (= moreResults "MORE_RESULTS_AFTER_LIMIT")
            not-finished-result? (= moreResults "NOT_FINISHED")
            more-available? (or more-after-limit-result? not-finished-result?)
            ;EMULATOR BEHAVES DIFFERENTLY AND CAUSES ENDLESS LOOP
            successive-empty-calls (if (and more-after-limit-result?
                                            (zero? (count entityResults)))
                                     (inc successive-empty-calls)
                                     0)
            empty-call-limit-reached? (>= successive-empty-calls 5)]

        (when instrument?
          (prn "=>>>>")
          (prn (str "-> empty call count: " successive-empty-calls))
          (prn (str "-> total results for call: " (count entityResults)))
          (prn (dissoc batch :entityResults)))

        (cond
          empty-call-limit-reached?
          combined-results

          (and has-limit-set? (>= (count combined-results) limit))
          combined-results

          (and has-limit-set? more-available?)
          (let [total-results (count combined-results)
                new-batch-size (min max-batch-size (- limit total-results))
                gql-query (str gql-query " LIMIT " new-batch-size)]
            (recur
             (run-query-next-page-request gql-query query-params endCursor)
             combined-results
             successive-empty-calls))

          more-available?
          (recur
           (run-query-next-page-request batched-gql-query query-params endCursor)
           combined-results
           successive-empty-calls)

          :else
          combined-results)))))


(comment
  (def result
    (run-query
     (api-client {:project-id "simply-prod"
                  :client
                  (simply.gcp.auth/keyfile-client "../keys/keyfile-prod.json")})
     "select * from `contract-search-fields` where field = \"UNDERWRITE\""
     {:ds/namespace-id "Contracts"
      :ds/project-id "simply-prod"}
     :limit 1
     :offset 1
     :instrument? true
     ))

  (count result)


  )


;; Base

(defn- wrap-datastore-params [project-id http-client]
  (fn [config]
    (cond-> config
        true (update :url #(format "https://datastore.googleapis.com/v1/projects/%s%s" project-id %))
        true (assoc :scope "https://www.googleapis.com/auth/datastore")
        (contains? config :body) (update :body json/generate-string)
        true http-client)))


(defn- wrap-error-handling [client]
  (fn [config]
    (let [{:keys [status body error opts] :as response} @(client config)]
      (when (or (nil? status) (<= 300 status) (> 200 status))
        (e/throw-app-error "Datastore Request Failed" {:status status
                                                       :body   body
                                                       :error  error
                                                       :url    (:url opts)
                                                       :method (:method opts)}))
      response)))


(defn api-client
  "wraps a simple.gcp.auth client with datastore parameters and error handling"
  [{:keys [project-id client]}]
  (->> client
       (wrap-datastore-params project-id)
       wrap-error-handling))
