(ns simply.validation.core
  (:require [simply.errors :as e]
            [simply.validation.rsa-id-number :as rsa-id-number]
            [clojure.string]))


(defn required-string? [s] (and (string? s) (not (empty? s))))


;;;; CELLPHONES

(defn valid-cellphone-number? [n]
  (boolean
   (and (string? n)
        (re-find #"^[0-9]*$" n)
        (= 10 (count n))
        (or (= "0" (first n)) (= \0 (first n))))))


(defn validate-cellphone-number
  "Returns a simply.errors/validation result"
  [cellphone-number]
  (if (true? (valid-cellphone-number? cellphone-number))
    (e/validation)
    (e/validation "Cellphone number should start with a zero and have a length of 10")))


;;;; EMAILS

(defn valid-email? [e]
  (boolean
   (and (string? e)
        (re-matches #"^((([!#$%&'*+\-/=?^_`{|}~\w])|([!#$%&'*+\-/=?^_`{|}~\w][!#$%&'*+\-/=?^_`{|}~\.\w]{0,}[!#$%&'*+\-/=?^_`{|}~\w]))[@]\w+([-.]\w+)*\.\w+([-.]\w+)*)$"
                    e))))


(defn validate-email
  "Returns a simply.errors/validation result"
  [e]
  (if (true? (valid-email? e))
    (e/validation)
    (e/validation "Email should be in the format foo@bar.com")))


;;;; ID NUMBERS

(defn valid-rsa-id-number? [id-number]
  (rsa-id-number/valid? id-number))


(defn validate-rsa-id-number
  "Returns a simply.errors/validation result"
  [id-number]
  (if (true? (valid-rsa-id-number? id-number))
    (e/validation)
    (e/validation "RSA Identity Number is invalid")))


(defn valid-passport-number? [passport] (required-string? passport))


(defn validate-passport-number
  "Returns a simply.errors/validation result"
  [passport]
  (if (true? (valid-passport-number? passport))
    (e/validation)
    (e/validation "Passport number is invalid")))


(defn validate-password
  "Validates if a password is of required length and has at least 1 number, special char, upper and lowercase letter"
  [password & {:keys [min-length max-length]
               :or {min-length 8 max-length 1000}}]
  (let [l (if (string? password) (count password) 0)
        not? #(nil? (re-matches % password))]
    (cond
      (> min-length l)
      (e/validation (str "Password must be at least " min-length " characters"))

      (> l max-length)
      (e/validation (str "Password must be no more than " min-length " characters"))

      :else
      (let [err
            (cond-> []
              (not? #".*[a-z].*")
              (conj "letter in small case")

              (not? #".*[A-Z].*")
              (conj "letter in capital case")

              (not? #".*[0-9].*")
              (conj "number")

              (not? #".*[^\w\d\s:]")
              (conj "special character"))]
        (if (empty? err)
          (e/validation)
          (e/validation
           (str "Password must contain at least 1 "
                (clojure.string/join ", " err))))))))


(defn valid-password?
  "see doc string for validate-password"
  [password & {:keys [min-length max-length]}]
  (:valid? (validate-password password :min-length min-length :max-length max-length)))
