(ns simply.validation.core
  (:require [simply.errors :as e]
            [simply.validation.rsa-id-number :as rsa-id-number]))


(defn required-string? [s] (and (string? s) (not (empty? s))))


;;;; CELLPHONES

(defn valid-cellphone-number? [n]
  (boolean
   (and (string? n)
        (re-find #"^[0-9]*$" n)
        (= 10 (count n))
        (or (= "0" (first n)) (= \0 (first n))))))


(defn validate-cellphone-number
  "Returns a simply.errors/validation result"
  [cellphone-number]
  (if (true? (valid-cellphone-number? cellphone-number))
    (e/validation)
    (e/validation "Cellphone number should start with a zero and have a length of 10")))


;;;; EMAILS

(defn valid-email? [e]
  (boolean
   (and (string? e)
        (re-matches #"^((([!#$%&'*+\-/=?^_`{|}~\w])|([!#$%&'*+\-/=?^_`{|}~\w][!#$%&'*+\-/=?^_`{|}~\.\w]{0,}[!#$%&'*+\-/=?^_`{|}~\w]))[@]\w+([-.]\w+)*\.\w+([-.]\w+)*)$"
                    e))))


(defn validate-email
  "Returns a simply.errors/validation result"
  [e]
  (if (true? (valid-email? e))
    (e/validation)
    (e/validation "Email should be in the format foo@bar.com")))


;;;; ID NUMBERS

(defn valid-rsa-id-number? [id-number]
  (rsa-id-number/valid? id-number))


(defn validate-rsa-id-number
  "Returns a simply.errors/validation result"
  [id-number]
  (if (true? (valid-rsa-id-number? id-number))
    (e/validation)
    (e/validation "RSA Identity Number is invalid")))


(defn valid-passport-number? [passport] (required-string? passport))


(defn validate-passport-number
  "Returns a simply.errors/validation result"
  [passport]
  (if (true? (valid-passport-number? passport))
    (e/validation)
    (e/validation "Passport number is invalid")))
