(ns simply.job-requests.cloud-scheduler
  (:require [simply.gcp.cloud-scheduler :as cs]
            [simply.job-requests.db :refer [JobsDb] :as db]
            [simply.deps :as s.deps]
            [clojure.string :as string]
            [simply.job-requests.core :as jr]
            [cheshire.core :as json]
            [taoensso.timbre :as log]
            [integrant.core :as ig]
            [simply.simply-core-config :as simply-core-config]))

(defn- job-name [{:keys [name service] :as job}]
  (str "__" service "__" name))

(defn- pubsub-job [{:keys [name schedule service] :as job}]
  (cs/job :name (job-name job)
          :description (str "Auto generated by " service)
          :schedule schedule
          :time-zone "Africa/Johannesburg"
          :target (cs/pubsub-target :project-id (s.deps/get-dep :gcp/project-id)
                                    :topic jr/job-request-topic-key
                                    :data (json/generate-string
                                           {:service service
                                            :jobname name}))))

(defn- extract-service [job]
  (let [core-name (last (string/split (:name job) #"/"))
        name-parts (string/split core-name #"__")]
    (if-not (>= (count name-parts) 3)
      (assoc job :name core-name)
      (assoc job
             :name (nth name-parts 2)
             :service (nth name-parts 1)))))

(defn- ->client []
  (cs/->client
   (s.deps/get-dep :gcp/project-id)
   (s.deps/get-dep :gcp/client)))

(defn db []
  (let [get-locations (memoize (fn [] (cs/get-all-locations (->client))))
        job-location (fn [job]
                       (let [locations       (set (get-locations))

                             location        (:location job)

                             valid-location? (contains? locations location)]

                         ;; If a valid location is set on the job, use it
                         ;; otherwise use the default location from simply.core config
                         (if valid-location?
                           location
                           (simply-core-config/cloud-scheduler-default-location))))]
    (reify JobsDb

      (list-jobs [_]
        (->> (cs/get-all-jobs-for-all-locations (->client))
             (map (fn [j]
                    (-> (select-keys j [:name :schedule :location])
                        extract-service)))))

      (create [_ job]
        (let [location (job-location job)
              pjob (pubsub-job job)
              response ((->client)
                        (cs/project->locations->list->jobs->create location pjob))]
          (log/info (str "Created Auto Job " (:name job) " with response " (:status response)))
          (cond
            (cs/success-response? response) :success

            (cs/already-exists-response? response)
            (do
              (log/error (str "Auto Job already exists: " (:name job)))
              :already-exists)

            :else (cs/throw-response response))))

      (delete [_ job]
        (let [location (job-location job)
              response ((->client)
                        (cs/project->locations->list->jobs->delete location (job-name job)))]
          (log/info (str "Deleted Auto Job " (:name job) " with response " (:status response)))
          (cond
            (cs/success-response? response) :success

            (cs/not-found-response? response)
            (do
              (log/error (str "Auto Job already removed: " (:name job)))
              :already-removed)

            :else (cs/throw-response response)))))))

(defmethod ig/init-key :simply.job-requests.cloud-scheduler/db [_ _]
  (db))

(comment
  (require 'simply.gcp.auth 'simply.gcp.keys 'simply.job-requests.core)

  (simply.gcp.auth/reset-credentials!)

  (let [project-id "simply-pre-prod"
        client (simply.gcp.auth/keyfile-client (simply.gcp.keys/key-file :simply-pre-prod))
        test-job (simply.job-requests.core/job "job-2" "1 0 * * *" "foo")]
    (with-redefs [simply.deps/get-dep (fn [k] (get {:gcp/project-id project-id
                                                    :gcp/client client} k))]
      #_(db/list-jobs (db))
      #_(db/create (db) test-job)
      #_(db/delete (db) (assoc test-job :location "europe-west1")))))
