(ns simply.gcp.datastore.entities
  (:require [clj-time.coerce :as time.coerce]
            [clj-time.format :as time.format]
            [clojure.spec.alpha :as s]
            [clojure.string :as string]
            [simply.errors :as e]
            [simply.gcp.util :as util]
            [clojure.data.codec.base64 :as b64]))

;;;; SPECS

(def allowed-entity-keys
  {:base
   "An entity is just a {} with some special keyword namespaced as ds defined below"
   :required
   {:ds/project-id "The project id"
    :ds/kind "The entity kind"}
   :optional
   {:ds/namespace-id "The namespace of the entity"
    :ds/indexes "A set of property keywords that you want to index"
    :ds/name "Entity named id"
    :ds/id "Entity Id"
    :ds/ancestor {:ds/kind "ancestor kind"
                  :ds/name "ancestor name"
                  :ds/id "ancestor id"}}})

(defn- not-empty-string? [s] (and (string? s) (seq s)))

(s/def :ds/string not-empty-string?)
(s/def :ds/project-id :ds/string)
(s/def :ds/kind :ds/string)
(s/def :ds/namespace-id :ds/string)
(s/def :ds/index-definition (s/or :keyword-index keyword?
                                  :vector-index (s/coll-of keyword? :kind vector?)))
(s/def :ds/indexes (s/coll-of :ds/index-definition :kind set?))
(s/def :ds/name :ds/string)
(s/def :ds/id integer?)
(s/def :ds/name-entity (s/keys :req [:ds/kind :ds/name]))
(s/def :ds/id-entity (s/keys :req [:ds/kind :ds/id]))
(s/def :ds/entity-key (s/or :name-entity :ds/name-entity
                            :id-entity :ds/id-entity))
(s/def :ds/ancestor :ds/entity-key)

(s/def :ds/entity (s/and :ds/entity-key
                         (s/keys :req [:ds/project-id
                                       :ds/namespace-id]
                                 :opt [:ds/indexes
                                       :ds/ancestor])))

(defn validate-entity? [e] (s/valid? :ds/entity e))

(defn- ds-namespace? [k] (string/starts-with? (str k) ":ds"))

(defn- ->datastore-name [k] (string/replace (str k) #"^:" ""))


;; PROPERTIES

;; see https://cloud.google.com/datastore/docs/reference/rest/v1/projects/runQuery#Value
(defprotocol DatastoreValue
  (prepare-value [this opts]))

;; Datastore indexes every property by default, this could
;; lead to problems, for example if the value of a string
;; property is too long for indexing. This library includes
;; no property in the index by default. If you like to index
;; a property include its key into the :ds/indexes entry of
;; the entity map:
;;
;; {:ds/kind "entity-key"
;;  :ds/name "123"
;;  :ds/indexes #{:name} ;; :name will be indexed
;;  :name "my entity"}
(defn- prepare-properties [entity opts]
  (into
    {}
    (keep
      (fn [[k v]]
        (let [list-value? (contains? (ancestors (type v)) java.util.List)]
          (when-not (ds-namespace? k)
            (let [opts          (update opts :path conj k) ;; The current path is used to determine if we should index
                  db-value      (prepare-value v opts)]
              [(->datastore-name k)
               (if list-value?
                 db-value
                 (assoc db-value
                        :excludeFromIndexes (not (contains? (:ds/indexes opts) (:path opts)))))]))))
      entity)))


(extend-protocol DatastoreValue
  (Class/forName "[B")
  (prepare-value [this opts]
    {:blobValue (String. (b64/encode this))})
  nil
  (prepare-value [this opts]
    {:nullValue this})
  Boolean
  (prepare-value [this opts]
    {:booleanValue this})
  Integer
  (prepare-value [this opts]
    {:integerValue this})
  clojure.lang.BigInt
  (prepare-value [this opts]
    {:integerValue this})
  Long
  (prepare-value [this opts]
    {:integerValue this})
  Float
  (prepare-value [this opts]
    {:doubleValue this})
  Double
  (prepare-value [this opts]
    {:doubleValue this})
  java.util.Date
  (prepare-value [this opts]
    {:timestampValue (util/unparse-zulu-date-format this)})
  java.util.UUID
  (prepare-value [this opts]
    {:stringValue (str this)})
  String
  (prepare-value [this opts]
    {:stringValue this})
  java.util.List
  (prepare-value [this opts]
    (let [opts (assoc opts :ds/indexes #{})] ;;We are not interested in indexing into arrays
      {:arrayValue {:values (map #(prepare-value % opts) this)}}))
  clojure.lang.PersistentArrayMap
  (prepare-value [this opts]
    {:entityValue {:properties (prepare-properties this opts)}})
  clojure.lang.PersistentHashMap
  (prepare-value [this opts]
    {:entityValue {:properties (prepare-properties this opts)}}))


;; KEYS

(defn prepare-partition [entity]
  (let [project {:projectId (:ds/project-id entity)}]
    (merge
     project
     (when-let [namespace-id (:ds/namespace-id entity)]
       {:namespaceId namespace-id}))))


(defn- prepare-entity-key [entity]
  (merge
   {:kind (:ds/kind entity)}
   (when-let [id (:ds/id entity)]
     {:id id})
   (when-let [name (:ds/name entity)]
     {:name name})))


;;https://cloud.google.com/datastore/docs/reference/rest/v1/Key
(defn prepare-key [entity]
  (let [key  (prepare-entity-key entity)
        path (if-let [ancestor (:ds/ancestor entity)]
               [(prepare-entity-key ancestor) key]
               [key])]
    {:partitionId (prepare-partition entity)
     :path        path}))


;; ENTITY

;;To facilitate indexes on nested maps, we allow the index definition to be either
;; * a keyword for backwards compatibility
;; * a vector representing a path to the property we want to index
;; For convenience in later lookup, we convert all keywords to a vector containing the keywords as this represents a top level path
(defn prepare-entity [entity]
  (let [opts {:ds/indexes (->> (:ds/indexes entity)
                               (map #(if (keyword? %) [%] %))
                               set)
              :path []}]
    {:key
     (prepare-key entity)
     :properties
     (prepare-properties entity opts)}))


;;;; PARSE ENTITIES

(declare parse-value parse-properties)


(def parse-fns
  {:stringValue :stringValue
   :integerValue #(Long/valueOf (:integerValue %))
   :booleanValue :booleanValue
   :nullValue (constantly nil)
   :doubleValue (fn [{:keys [doubleValue]}]
                  (if (number? doubleValue)
                    doubleValue
                    (Double/valueOf doubleValue)))
   :timestampValue #(->> (:timestampValue %)
                         util/unparse-zulu-date-format
                         (time.format/parse (:date-time time.format/formatters))
                         time.coerce/to-date)
   :blobValue #(b64/decode (.getBytes (:blobValue %)))
   :arrayValue #(map parse-value (get-in % [:arrayValue :values]))
   :entityValue #(parse-properties (get-in % [:entityValue :properties]))})


(defn- parse-value [value]
  (if-let [parse-fn (some
                     (fn [[k f]]
                       (when (contains? value k)
                         f))
                     parse-fns)]
    (parse-fn value)
    (e/throw-app-error "Could Not Parse Entity Value" {:value value})))


(defn- parse-properties [properties-response]
  (into
   {}
   (map
    (fn [[k value-response]]
      [(keyword k) (parse-value value-response)])
    properties-response)))


(defn- parse-entity-path [path]
  (merge
   {:ds/kind (:kind path)}
   (when-let [id (:id path)]
     {:ds/id (Long/parseLong id)})
   (when-let [name (:name path)]
     {:ds/name name})))


(defn- parse-key [entity-response]
  (let [key (get-in entity-response [:entity :key])
        has-ancestor? (> (count (:path key)) 1)
        path (last (:path key))]
    (assert key)
    (merge
     (parse-entity-path path)
     (when-let [namespace-id (get-in key [:partitionId :namespaceId])]
       {:ds/namespace-id namespace-id})
     (when has-ancestor?
       {:ds/ancestor (parse-entity-path (first (:path key)))}))))


(defn parse-index-info [properties-response]
  {:ds/indexes (set
                (keep
                 (fn [[k value]]
                   (when-not (:excludeFromIndexes value)
                     k))
                 properties-response))})


(defn parse-entity [entity-response]
  (let [properties-response (:properties (:entity entity-response))]
    (->
     (merge
      (parse-properties properties-response)
      (parse-index-info properties-response)
      (parse-key entity-response))
     (dissoc :meta-upsert-at))))
