(ns simply.banking-details.banks
  (:require #?(:clj [clojure.string :as string])
            #?(:clj [clojure.java.io :as io])
            [simply.banking-details.branches :refer [branch-code-bank-id-map]]))

(defn- default-short-names [b]
  (assoc b :short-name (:short-name b (:bank-name b))))


(def banks
  (->>
   [{:bank-id        "3",
     :bank-code      "16",
     :bank-name      "ABSA BANK",
     :short-name     "ABSA"
     :validate-ind   -1,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code "f"}
    {:bank-id        "4",
     :bank-code      "1",
     :bank-name      "STANDARD BANK OF SOUTH AFRICA LTD",
     :short-name     "Standard Bank"
     :validate-ind   -1,
     :weighting      "11987654321",
     :fudge-factor   0,
     :modulus        11,
     :exception-code "m"}
    {:bank-id        "5",
     :bank-code      "2",
     :bank-name      "NEDBANK LIMITED",
     :short-name     "Nedbank"
     :validate-ind   -1,
     :weighting      "11987654321",
     :fudge-factor   nil,
     :modulus        11,
     :exception-code "1"}
    {:bank-id        "6",
     :bank-code      "3",
     :bank-name      "FIRSTRAND BANK",
     :short-name     "FNB/RMB"
     :validate-ind   -1,
     :weighting      "12121212121",
     :fudge-factor   0,
     :modulus        10,
     :exception-code nil}
    {:bank-id        "7",
     :bank-code      "6",
     :bank-name      "BANK OF ATHENS SA LTD",
     :short-name     "Bank of Athens"
     :validate-ind   -1,
     :weighting      "0",
     :fudge-factor   0,
     :modulus        11,
     :exception-code nil}
    {:bank-id        "8",
     :bank-code      "7",
     :bank-name      "AFRICAN BANK LIMITED",
     :short-name     "African Bank"
     :validate-ind   -1,
     :weighting      "12121212121",
     :fudge-factor   0,
     :modulus        10,
     :exception-code nil}
    {:bank-id        "9",
     :bank-code      "9",
     :bank-name      "MERCANTILE BANK LIMITED",
     :validate-ind   -1,
     :weighting      "1A987654321",
     :fudge-factor   0,
     :modulus        11,
     :exception-code "b"}
    {:bank-id        "10",
     :bank-code      "10",
     :bank-name      "CAPITEC BANK LIMITED",
     :short-name     "Capitec Bank"
     :validate-ind   -1,
     :weighting      "21987654321",
     :fudge-factor   0,
     :modulus        11,
     :exception-code nil}
    {:bank-id        "11",
     :bank-code      "13",
     :bank-name      "SOUTH AFRICAN POST OFFICE",
     :short-name     "SA Post Bank"
     :validate-ind   -1,
     :weighting      "42184218421",
     :fudge-factor   0,
     :modulus        10,
     :exception-code nil}
    {:bank-id        "12",
     :bank-code      "14",
     :bank-name      "BANK WINDHOEK BEPERK",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "13",
     :bank-code      "17",
     :bank-name      "STANDARD BANK SWAZILAND LTD",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "14",
     :bank-code      "19",
     :bank-name      "TEBA BANK LIMITED",
     :validate-ind   -1,
     :weighting      nil,
     :fudge-factor   0,
     :modulus        11,
     :exception-code "2"}
    {:bank-id        "15",
     :bank-code      "22",
     :bank-name      "HABIB OVERSEAS BANK LIMITED",
     :validate-ind   -1,
     :weighting      "7654321",
     :fudge-factor   0,
     :modulus        11,
     :exception-code "j"}
    {:bank-id        "16",
     :bank-code      "23",
     :bank-name      "LESOTHO BANK LIMITED",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "17",
     :bank-code      "27",
     :bank-name      "FBC FIDELITY BANK LTD",
     :validate-ind   -1,
     :weighting      "13579135791",
     :fudge-factor   0,
     :modulus        10,
     :exception-code nil}
    {:bank-id        "18",
     :bank-code      "28",
     :bank-name      "UNIBANK LIMITED",
     :validate-ind   -1,
     :weighting      "0",
     :fudge-factor   0,
     :modulus        0,
     :exception-code nil}
    {:bank-id        "19",
     :bank-code      "29",
     :bank-name      "ABN AMRO BANK",
     :validate-ind   -1,
     :weighting      "987654321",
     :fudge-factor   0,
     :modulus        11,
     :exception-code nil}
    {:bank-id        "20",
     :bank-code      "33",
     :bank-name      "CITIBANK",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "21",
     :bank-code      "34",
     :bank-name      "BARCLAYS BANK PLC SA BRANCH",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "22",
     :bank-code      "36",
     :bank-name      "HBZ BANK LIMITED",
     :validate-ind   -1,
     :weighting      "137131",
     :fudge-factor   0,
     :modulus        11,
     :exception-code "d"}
    {:bank-id        "23",
     :bank-code      "37",
     :bank-name      "NEDBANK SWAZILAND LIMITED",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "24",
     :bank-code      "38",
     :bank-name      "NEDBANK LESOTHO LIMITED",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "25",
     :bank-code      "39",
     :bank-name      "INVESTEC BANK LIMITED",
     :short-name     "Investec"
     :validate-ind   -1,
     :weighting      "000NJHD7531",
     :fudge-factor   0,
     :modulus        11,
     :exception-code nil}
    {:bank-id        "26",
     :bank-code      "42",
     :bank-name      "STANDARD CHARTERED BANK SA",
     :validate-ind   -1,
     :weighting      "27654321000",
     :fudge-factor   0,
     :modulus        11,
     :exception-code nil}
    {:bank-id        "27",
     :bank-code      "SAHL",
     :bank-name      "SA HOME LOANS",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "28",
     :bank-code      "43",
     :bank-name      "NEDBANK NAMIBIA",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "29",
     :bank-code      "44",
     :bank-name      "BIDVEST BANK LIMITED",
     :short-name     "Bidvest Bank"
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "30",
     :bank-code      "45",
     :bank-name      "SOCIETE GENERAL JHB BRANCH",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "31",
     :bank-code      "46",
     :bank-name      "STANDARD BANK LESOTHO",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "32",
     :bank-code      "12",
     :bank-name      "SA RESERVE BANK",
     :validate-ind   -1,
     :weighting      "11187654321",
     :fudge-factor   0,
     :modulus        11,
     :exception-code nil}
    {:bank-id        "33",
     :bank-code      "11",
     :bank-name      "MTN BANKING (STANDARD BANK)",
     :validate-ind   -1,
     :weighting      "13971397131",
     :fudge-factor   0,
     :modulus        10,
     :exception-code nil}
    {:bank-id        "34",
     :bank-code      "5",
     :bank-name      "PEOPLE BANK LTD INC. PEP BANK",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "35",
     :bank-code      "8",
     :bank-name      "NEDBANK LTD INC. BOE BANK",
     :validate-ind   -1,
     :weighting      "11987654321",
     :fudge-factor   nil,
     :modulus        11,
     :exception-code "1"}
    {:bank-id        "36",
     :bank-code      "15",
     :bank-name      "ABSA (FOR SAMOS USE ONLY)",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "37",
     :bank-code      "21",
     :bank-name      "(EX - SWA STILL ON VISA)",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "38",
     :bank-code      "24",
     :bank-name      "PEOPLES BANK LTD INC. NBS",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "39",
     :bank-code      "26",
     :bank-name      "PERMANENT BANK",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "40",
     :bank-code      "30",
     :bank-name      "ABSA - ITHALA",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "41",
     :bank-code      "31",
     :bank-name      "(EX - SDSB SWAZI)",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "42",
     :bank-code      "32",
     :bank-name      "(EX - COMMUNITY)",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code nil}
    {:bank-id        "43",
     :bank-code      "47",
     :bank-name      "GRINDROD BANK LIMITED",
     :short-name     "Gindrod Bank"
     :validate-ind   -1,
     :weighting      "849TNJHD731",
     :fudge-factor   0,
     :modulus        11,
     :exception-code "k"}
    {:bank-id        "44",
     :bank-code      "0",
     :bank-name      "ALBARAKA BANK LIMITED"
     :short-name     "Albaraka Bank",
     :validate-ind   0,
     :weighting      nil,
     :fudge-factor   0,
     :modulus        0,
     :exception-code nil}
    {:bank-id        "99",
     :bank-code      "999",
     :bank-name      "TYME BANK LIMITED",
     :short-name     "Tyme Bank",
     :validate-ind   -1,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code "f"}
    {:bank-id        "98",
     :bank-code      "998",
     :bank-name      "DISCOVERY BANK LIMITED",
     :short-name     "Discovery Bank"
     :validate-ind   -1,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code "f"}
    {:bank-id        "97",
     :bank-code      "997",
     :bank-name      "BANK ZERO MUTUAL BANK",
     :short-name     "Bank Zero",
     :validate-ind   -1,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code "f"}
    {:bank-id        "96",
     :bank-code      "996",
     :bank-name      "SASFIN BANK LIMITED",
     :short-name     "Sasfin Bank",
     :validate-ind   -1,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code "f"}
    {:bank-id        "94",
     :bank-code      "994",
     :bank-name      "JP Morgan Chase Bank",
     :short-name     "JP Morgan Chase Bank",
     :validate-ind   -1,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code "f"}
    {:bank-id        "17",
     :bank-code      "177",
     :bank-name      "Capitec Business",
     :short-name     "Capitec Business"
     :validate-ind   -1,
     :weighting      nil,
     :fudge-factor   nil,
     :modulus        nil,
     :exception-code "f"}]
   (map default-short-names)
   vec))


(def banks-map (->> banks
                    (map (juxt :bank-id identity))
                    (into {})))


(defn bank-from-branch-code [branch-code]
  (let [bank-id (get branch-code-bank-id-map branch-code)
        bank (get banks-map bank-id)]
    (when bank
      (assoc bank :branch-code branch-code))))


(def default-branch-codes [{:name "ABSA" :branch-code "632005"}
                           {:name "FNB/RMB" :branch-code "250655"}
                           {:name "Standard Bank" :branch-code "051001"}
                           {:name "Nedbank" :branch-code "198765"}
                           {:name "Capitec Bank" :branch-code "470010"}
                           {:name "African Bank" :branch-code "430000"}
                           {:name "Bank of Athens" :branch-code "410506"}
                           {:name "Bank Zero" :branch-code "888000"}
                           {:name "Bidvest Bank" :branch-code "462005"}
                           {:name "Discovery Bank" :branch-code "679000"}
                           {:name "Investec" :branch-code "580105"}
                           {:name "SA Post Bank" :branch-code "460005"}
                           {:name "Sasfin Bank" :branch-code "683000"}
                           {:name "Tyme Bank" :branch-code "678910"}
                           {:name "Capitec Business" :branch-code "450105"}])


(def bank-code-bank-account-groups-map
  {"1" :standard-bank
   "29" :investec-bank-group
   "7"  :investec-bank-group
   "39" :investec-bank-group
   "47" :investec-bank-group
   "13" :investec-bank-group
   "12" :investec-bank-group
   "11" :investec-bank-group
   "42" :investec-bank-group
   "19" :investec-bank-group
   "16" :absa
   "10" :mercantile-bank-group
   "9"  :mercantile-bank-group
   "3" :nedbank-group
   "22" :nedbank-group
   "2" :nedbank-group
   "8" :nedbank-group
   "28" :nedbank-group
   "36" :hbz-group
   "6" :sa-bank-of-athens
   "999" :tyme-bank
   "998" :discovery-bank
   "997" :bank-zero
   "996" :sasfin-bank
   "995" :african-bank
   "994" :jp-morgan
   "177" :capitec-business})

;;;; PARSING

#?(:clj
   (defn- parse-csv [fields delimiter path]
     (let [rows        (->> path
                            slurp
                            string/split-lines
                            rest
                            (map #(string/split % (re-pattern delimiter))))
           fields-list (->> (count rows)
                            range
                            (map (constantly fields)))]
       (->> (interleave fields-list rows)
            (partition 2)
            (map (fn [[k v]]
                   (->> (interleave k v)
                        (partition 2)
                        (map vec)
                        (into {}))))))))

#?(:clj (defn- null? [x] (= "NULL" x)))

#?(:clj (defn- set-nil [x t]
          (if (null? x) nil (t x))))

#?(:clj
   (defn- parse-banks-csv
     "Parses the banks csv file received from iua. expects fields
      BankId;BankCode;BankName;ValidateInd;Weighting;FudgeFactor;Modulus;ExceptionCode

      Copy the output to (def banks [])"
     []
     (let [delimiter ";"
           fields [:bank-id :bank-code :bank-name
                   :validate-ind :weighting :fudge-factor
                   :modulus :exception-code]
           path "src/simply/banking_details/data/Banks.csv"]
       (->> (parse-csv fields delimiter path)
            (map
             (fn [b]
               (-> b
                   (update :validate-ind read-string)
                   (update :weighting set-nil identity)
                   (update :fudge-factor set-nil read-string)
                   (update :modulus set-nil read-string)
                   (update :exception-code set-nil identity))))))))


#?(:clj
   (defn- parse-brances-csv
     "Parses the banks csv file received from iua. expects fields
      BankId;BankCode;BankName;ValidateInd;Weighting;FudgeFactor;Modulus;ExceptionCode
      Copy the output to simply.banking-details.branches/branch-code-bank-id-map"
     []
     (let [delimiter ";"
           fields [:bank-branch-id :bank-id :branch-code :branch-name]
           path "src/simply/banking_details/data/Branches.csv"]
       (set! *print-length* 6000)
       (->> (parse-csv fields delimiter path)
            (map (fn [r]
                   (update r :branch-code #(->> % read-string (format "%06d")))))
            (map (juxt :branch-code :bank-id))
            (into {})))))
