(ns simply.dev
  (:require [overtone.at-at :as at]
            [simply.cqrs :refer [CQRS CqrsSystem] :as cqrs]
            [taoensso.timbre :as log]
            [integrant.core :as ig]))


(defonce cqrs-pool (at/mk-pool))


(defn handle-topic [topic data options]
  (let [topic-subscriptions (->> (methods cqrs/handle-topic)
                                 keys
                                 (filter coll?)
                                 (group-by first))]
    (when-let [subs (get topic-subscriptions topic)]
      (doseq [[topic-key sub-key] subs]
        (cqrs/handle-topic topic-key sub-key data options)))))


(defn- dev-cqrs [get-options]
  (reify
    CQRS

    (require-actions [this actions]
      (cqrs/require-actions this :required-actions actions))
    (require-actions [this _ actions]
      (let [options (assoc (get-options) :cqrs this)]
        (at/after
         1000
         (fn []
           (doseq [action actions]
             (try (cqrs/action-handler options action)
                  (catch Exception e
                    (log/error e (format "Could not processes action %s" (:type action)))))
             ))
         cqrs-pool)))

    (wrap-event [this event] event)

    (notify [this topic coll]
      (if (= topic cqrs/event-topic)
        (doseq [{:keys [type] :as event} coll]
          (log/infof "Event => %s" type))
        (do
          (log/infof "Enqueue => %s (%s)" topic (count coll))
          (at/after
           1000
           (fn []
             (doseq [data coll]
               (try (handle-topic topic data (get-options))
                    (catch Exception e
                      (log/error e (format "Could not processes subscription %s" topic))))
               ))
           cqrs-pool)
          )))))


(defn dev-cqrs-system [options]
  (let [*options (atom options)
        get-options (fn [] @*options)
        command-handler (fn [type data user] (cqrs/command-handler (get-options) type data user))
        action-handler (fn [action] (cqrs/action-handler (get-options) action))
        cqrs (dev-cqrs get-options)
        system (reify
                 CqrsSystem
                 (get-command-handler [this] command-handler)
                 (get-action-handler [this] action-handler)
                 (send-messages [this topic coll] (cqrs/notify cqrs topic coll))
                 (request-actions [this actions]
                   (cqrs/require-actions cqrs actions))
                 (start [this] nil)
                 (stop [this] nil))]
    (swap! *options (fn [o] (assoc o :cqrs cqrs :cqrs-system system)))
    system))


(defmethod ig/init-key :simply.dev/cqrs-system
  [_ {:keys [dependencies] :as options}]
  (dev-cqrs-system (merge (dissoc options :dependecies)
                          dependencies)))

(defmethod cqrs/handle-command :test-command [t data options]
  (log/infof "***TEST Action: %s %s" t data)
  (cqrs/command-result
    [(cqrs/event :test-command-event {})]
    [(cqrs/action :test-action {:foo "action"})
     (cqrs/action :test-chunked-action [{:a "1"} {:b "2"}])]))


(defmethod cqrs/handle-action :test-action [t data options]
  (log/infof "***TEST Action: %s %s" t data)
  (cqrs/action-result
   [(cqrs/event :test-action-event {})]))


(defmethod cqrs/handle-action :test-chunked-action [t coll options]
  (cqrs/chunk-one-at-a-time (:cqrs options) t coll
                (fn []
                  (log/infof "***TEST Chunked Action: %s" (first coll)))))


(defmethod cqrs/handle-command :test-failing-action-command [t data options]
  (log/infof "***TEST Failing action command: %s %s" t data)
  (cqrs/command-result
    [(cqrs/event :test-failing-action-event {})]
    [(cqrs/action :test-failing-action data)]))


(defmethod cqrs/handle-action :test-failing-action [t data options]
  (assert false))
