# simply.core

[![Clojars Project](https://img.shields.io/clojars/v/za.co.simply/simply.core.svg)](https://clojars.org/za.co.simply/simply.core)

This Library represents various functions and tools reused across Simply's clojure apps.

Many of the bigger components (like those related to google cloud project) have companion **[Integrant](https://github.com/weavejester/integrant)** `init-key` multimethods for easy consumption with Integrant. [dyn-edn](https://github.com/walmartlabs/dyn-edn) is recommended as it provides flexibility in your system.edn file.

> **NOTE**: You do not have to use Integrant and all functions are available for direct consumption.

### a note on config

A basic config.clj file using `integrant` might look like this. This makes it easy for your app and repl to consume and change the `system.edn` config as well as makes your env config injectable.

```
(ns your-app.config
  (:require [clojure.java.io :as io]
            [com.walmartlabs.dyn-edn :as dyn-edn]
            [integrant.core :as ig]))


(defn system []
  (ig/read-string {:readers (dyn-edn/env-readers)}
                  (slurp (io/resource "system.edn"))))


(defmethod ig/init-key :your-app/config [_ m] m)
```

```
;; Example system.edn
:suffix/config
 {:version #dyn/prop [CI_VERSION "development"]
  :secure? true
  :project-id #dyn/prop [GCP_PROJECT_ID "dogmatix"]
  :api-key  #dyn/prop [GCP_API_KEY ""]
  :container-name #dyn/prop [CONTAINER_NAME "suffix-2"]
  :container-id #dyn/prop [CONTAINER_ID "suffix-2"]
  :tag-id #dyn/prop [GOOGLE_TAG_MANAGER_ID nil]
  :log-level :info}}
```

### a note on development

Integrant makes reloaded development a breeze. See [https://github.com/weavejester/integrant-repl](https://github.com/weavejester/integrant-repl) for reloaded workflow.

An example user.clj file can be seen below. This examples allows for switching between a system with an in memory database and one with a datastore database.

```
(ns user
  (:require [your-app.config :as config]
            [integrant.repl :refer [clear go halt prep init reset reset-all set-prep!]]))


(defn dev-system []
  (-> (config/system)
      (assoc-in [:simply.server/jetty :port] 3467)
      (assoc-in [:your-app.server/web :proxy] false)
      (assoc-in [:your-app.server/handler :log-appenders] [])
      (assoc-in [:your-app/db :db-provider :key] :your-app.integration/memorydb)))


(defn datastore-system []
  (-> (dev-system)
      (assoc-in [:your-app/config :project-id] "simply-pre-prod")
      (assoc-in [:your-app/db :db-provider :key] :your-app.integration.gcp.datastore/contracts-db)
      (assoc-in [:your-app.integration.gcp.datastore/contracts-db :client :key]
                :simply.gcp.auth/keyfile-client)))


(defn reload-config
  ([]
   (reload-config dev-system))
  ([system]
   (reset)
   (set-prep! system)
   (prep)
   (go)))


(defn reload-datastore-config [] (reload-config datastore-system))


(ig/load-namespaces (dev-system))
(set-prep! dev-system)
```
---

# Server

## simply.cqrs

### Overview

**Commands**

Any change to data should be triggered via a command. A command typically does 1 side effect (like saving data) and then returns a `cqrs/command-result`. The command result exists of a list of `cqrs/event` (at least 1 is required), a list of `cqrs/action` (0 or more) and optional data that is returned to the caller.

**Event**

A `cqrs/event` is a map that consists of a type and a payload. Events are pushed to the `events` topic to be processed by some other service

In Production, all Simply events are pushed to GCP pubsub and consumed by the `vitalstatistix` service and GCP Dataflow.

**Action**

A `cqrs/action` represents a set of side effects triggered by command handlers, but does not need to happen in the command handlers transaction. For example, a command can specify that an email needs to be sent, but the sending of the email will be done by the action and not by the command. This allows for better decoupled code as well as easier testing as actions are just data. Actions are analogous to Domain Driven Design's Domain-Events.

### Implementation

**CQRS Protocol**

This protocol defines how your system will deal with actions and events after a command has been handled.

* **require-actions** Should push actions into a mechanism that will handle them.
* **wrap-event** Can be used to wrap additional data into the event.
* **notify** Pushes messages to a defined topic. You will define your actions topic. Events will be pushed to `events` topic.

The protocol allows `simply.cqrs` to define _what_ should happen but not _how_ it happens. This allows you to run `simply.cqrs` in different environments.

**CqrsSystem Protocol**

A protocol that can be used to implement a cqrs system.

* **get-command-handler** Should return a fn that accepts a type and data.
* **get-action-handler** Should return a fn that accepts a `simply.cqrs/action`.
* **start** Starts the system.
* **stop** Stops the system.

**Environments**

* **Dev** `simply.dev` provides `dev-cqrs-system` and `ig/init-key :simply.dev/cqrs-system`, implementing `CqrsSystem` that simply pushes actions directly to the action handler after a delay and logs events to console.
* **GCP Pubsub** `simply.gcp.pubsub.cqrs` provides `pubsub-cqrs-system` and `ig/init-key :simply.gcp.pubsub.cqrs/pubsub-cqrs-system`, implementing `CqrsSystem` that uses GCP Pubsub for all required-actions, events and notification handling.
* **Testing** You can implement your own CQRS for testing purposes by either implementing `CQRS` or `CqrsSystem` based on your needs.

**Commands**

You register commands by extending the `cqrs/handle-command` multimethod. These methods return a `cqrs/command-result`. A result includes an events collection (1 or more), an actions collection (0 or more) and optional data.

You dispatch commands by calling `cqrs/command-handler`. The command handler needs a `:cqrs` key that implements the `CQRS` protocol as part of the options. The command handler will dispatch to `cqrs/handle-command`, validate resulting events and actions, notify of any events and notify of any required actions.

`ig/init-key :simply.cqrs/command-handler` will return a command-handler that only takes a type and data with the options injected.

**Events**

Use `cqrs/event` to create events.

**Actions**

Use `cqrs/action` to create actions.

You register actions by extending the `cqrs/handle-action` multimethod. These methods return a `cqrs/action-result`. A result includes an events collection (0 or more).

You can dispatch actions by calling `cqrs/action-handler`. The action handler needs a `:cqrs` key that implements the `CQRS` protocol as part of the options. the action handler will dispatch to `cqrs/handle-action`, validate the resulting events and notify of any events.
Typically the `require-actions` function on the `CQRS` protocol will typically be responsible for triggering the call to `cqrs/action-handler`.

`ig/init-key :simply.cqrs/action-handler` can also be used to dispatch actions. It will return a function that takes an action with the options injected.

**Multiple Actions**

In some cases you want to batch actions together. You can use the `requeue-action-chunks-if-needed` (or `chunk-events`, `chunk-entities`, `chunk-one-at-a-time`) to break up an action with a collection as its data, into more actions with smaller collection sizes.

Example, we can only save 20 entities at a time, but a certain action requires us to save 50. The command handler can still dispatch 50 entities as one action, but we can have the system break that up into 3 actions dispatching 20,20 and 10 entities at a time (see below for an example).

**Example**

Below is an example of how the cqrs library might be used in conjunction with Integrant. It uses the `simply.dev/dev_cqrs` implementation of the `CQRS` protocol. Obviously this implementation can easily be swapped out in the `system.edn` file.

```
;; system.edn

 :simply.dev/cqrs-system
 {}

 :your-app/handle-command
 {:cqrs-system #ig/ref :simply.dev/cqrs-system

 :your/app
 {:handle-command #ig/ref :your-app/handle-command}

;; ig key registration

(defmethod ig/init-key :your-app/handle-command [_ {:keys [cqrs-system]}]
  (cqrs/get-command-handler cqrs-system))

;; COMMANDS

 (defmethod cqrs/handle-command :test-command [t data options]
  (log/info "This is a test Command")
  (cqrs/command-result
   [(cqrs/event :test-command-event {})] ;; we return a single event
   [(cqrs/action :test-action {:foo "action"}) ;; we return 2 actions, one will be chunked
    (cqrs/action :test-chunked-action [{:a "1"} {:b "2"}])])
  )


;; ACTIONS

(defmethod cqrs/handle-action :test-action [t data options]
  (log/info "This is a test action that will return a single event")
  (cqrs/action-result
   [(cqrs/event :test-action-event {})])
  )


(defmethod cqrs/handle-action :test-chunked-action [t coll options]
  (cqrs/chunk-one-at-a-time
    (:cqrs options) t coll
    (fn []
      (log/info "This is a test action that will chunk it's collection into lists of 1))))


;; EXECUTION

(defmethod ig/init-key :your/app
  [_ {:keys [handle-command]}]
  (handle-command :test-command {:some "data}))
```

## simply.server

**ig/init-key :simply.server/jetty** implements all Integrant start, stop and reload multimethods.

```
;; Example system.edn

:simply.server/jetty
 {:port 8080
  :handler #ig/ref :your-app/handler}
```

## simply.web

* **google-tag-manager-scripts** Generates Google Tag Manager scripts if `GOOGLE_TAG_MANAGER_ID` env variable is available
* **google-cloud-project-routes** Compojure routes for google health checks
* **debug-routes** Routes for checking versions and testing exceptions

## simply.errors

Functions for dealing with errors. Relies on `slingshot`

## simply.scheduling.core

### Interval Jobs

An interval job is built to be scheduled on many services simultaneously, but only run on one instance at a time. This allows stateless services to define and run jobs without the need to have a single job running service. All server instances will check a `ScheduleDb` (that is shared by all services) to determine if a job is allowed to run. The scheduler will check the `ScheduleDb` based on the jobs `check-interval`. If the interval time has elapsed (as determined by the `ScheduleDb`) the interval job will run.

**interval-job**

Defines an interval job.

Takes a `job-key` string, an `interval` in milliseconds and a `job-f` that will run when the interval has elapsed. `job-f` takes no parameters

By default the `check-interval` is a 4th of the job interval. The `check-interval` can be changed by calling `with-check-interval`.

**start-interval-job**

Initiates a job schedule and returns a function that stops the scheduler when executed.

The scheduler has a delay of 10 seconds before the first job will run

**ScheduleDb Protocol**

`simply.scheduling.db` provides the following implementations of `ScheduleDb`. Each have a companion `ig/init` implementation at `simply.scheduling.db/<function-name>`

* **datastore-schedule-db** uses GCP datastore
* **never-run-schedule-db** never runs a job
* **in-memory-schedule-db** used for dev purposes on a local machine when only 1 server is running


**Example**

```
(let [job-key  "my-job"
      interval 10000
      job-f    #(prn "** WHOOP **")
      job      (interval-job job-key interval job-f)
      stop-job (start-interval-job (in-memory-schedule-db) job)]
  (prn "Job will start in 10 seconds and run for 25 seconds")
  (Thread/sleep 35000)
  (stop-job)
  (prn "Job Stopped"))

```

---

# Google Cloud Project

### Suggested Integrant system.edn for gcp config

```
 :your-app/config
 {:version #dyn/prop [CI_VERSION "development"]
  :project-id #dyn/prop [GCP_PROJECT_ID "dogmatix"]
  :api-key  #dyn/prop [GCP_API_KEY ""]
  :container-name #dyn/prop [CONTAINER_NAME "suffix-2"]
  :container-id #dyn/prop [CONTAINER_ID "suffix-2"]}}

```

## simply.gcp.auth

Different http clients to use with Google Cloud Project API's. These clients are typically passed to the Simply implementations of the various API's

* **emulator-client (ig/init-key :simply.gcp.auth/emulator-client)**
	* for use with datastore and pubsub emulators that run locally
* **keyfile-client (ig/init-key :simply.gcp.auth/keyfile-client)**
	* used to access GCP api's while authenticating using a keyfile
	* requires `keyfile-path` param
* **default-credentials-client (ig/init-key :simply.gcp.auth/default-credentials-client)**
	* used by containers running on google cloud that have access to default credentials

```
;; system.edn example

:simply.gcp.auth/keyfile-client
 {:keyfile-path "./keyfile-preprod.json"}

:simply.gcp.auth/default-credentials-client {}

:simply.gcp.auth/emulator-client {}
```

## simply.gcp.errorreporting

A `taoensso/timbre` appender for GCP error reporting

**ig/init-key :simply.gcp.errorreporting/appender** returns `{:name :clouderrors :appender {...}}`
and requires `project-id`, `api-key`, `container-name`, `container-id`

```
;; system.edn example

:your-app/handler
 {:log-appenders [#ig/ref :simply.gcp.errorreporting/appender]}

:simply.gcp.errorreporting/appender
 {:config #ig/ref :your-namespace/config}

:your-app/config
 {:log-level :info
  ...}
```
```
;; example your-app/handler

(defmethod ig/init-key :your-app/handler
  [_ {:keys [log-appenders config] :as options}]
  (logger/merge-config!
   {:level (:log-level config)
    :appenders (->> log-appenders
                    (map (juxt :name :appender))
                    (into {}))})
  (-> (routes ...))
```

## simply.gcp.datastore.core

A client to access GCP datastore.

**Getting started:**

* Create an auth client from `simply.gcp.auth`
* Wrap your auth client using `api-client`
* Create Entity and query wrappers for your project-id and namespaces using `wrap-entity` and `wrap-query-params`

```
;; Example usage

(ns your-app.datastore
  (:require [simply.gcp.datastore.core :as api]
            [integrant.core :as ig]))


(defprotocol Database
  (get-contracts [this id]))


(defn fetch-contracts [client contracts-query id]
  (->> (api/run-query
        client
        (str "SELECT * FROM contract where id = \"" id "\"")
        (contracts-query api/weak-read-option))
       (map api/domain-entity)))


(defn datastore [{:keys [client project-id] :as options}]
  (let [client (api/api-client options)
        contracts-namespace "Contracts"
        contracts-query (api/wrap-query-params project-id contracts-namespace)]
    (reify
      Database
      (get-contracts [this id]
        (fetch-contracts client contracts-query id)))))


(defmethod ig/init-key :your-app/datastore
  [_ {:keys [client config] :as options}]
  (datastore (assoc config :client client)))
```
```
;; Example system.edn

:your-app/datastore
 {:config #ig/ref :your-app/config
  :client #ig/ref :simply.gcp.auth/default-credentials-client}

:your-app/config
 {:project-id #dyn/prop [GCP_PROJECT_ID "dogmatix"]
  ...}
```

**Entities**

When *creating* datastore entities you need to:

* provide a `:ds/kind` String that represents the entity kind
* provide either a `:ds/name` String or a `:ds/id` Int. This along with `:ds/kind` will build the entity key
* provide a `:ds/indexes` Set of keywords. Only the columns listed in this set will be indexed by datastore
* wrap your entity with the function created by `wrap-entity` to ensure the namespace and project-id is set
* dates should be stored as `java.util.Date` objects
* keywords can be stored as strings and converted
* uuid's should be stored as strings

```
(defn person-entity [{:keys [id person-type] :as person}]
 (let [id (.toString id)]
  (merge
   person
   {:ds/kind "people"
    :ds/name id
    :ds/indexes #{:id :name :surname :person-type}
    :id id
    :person-type (name person-type)})))

;; lets create a person entity
(let [wrap-admin-entities (wrap-entity "project-1" "admin")
      person         {....}
      client         (api-client {:project-id "project-1"
                                  :client
                                   (simply.gcp.auth/default-credentials-client)})]
  (->> person
       person-entity
       wrap-admin-entities
       (upsert-entity client)
```

When _receiving_ entities from the db

* it is important to use `unwrap-entity` to remove any datastore spesific keys.
* you can use `domain-entity` instead of `unwrap-entity`. This will return kebab-case keywords. This is handy if you want to maintain code conventions when your datastore entities use a different conventions.
* You should also be conscious of keywords, uuids and dates and that you convert these to the desired types if needed.

```
(defn person [{:keys [id person-type date-of-birth] :as entity}]
   (-> entity
       domain-entity
       (merge
        {:id (java.util.UUID/fromString id)
         :person-type (keyword person-type
         :timestamp (.getTime date-of-birth})))

;; lets fetch a person
(let [person-key     (name-key "people" "213jo832-32kh42-fake-uuid")
      client         (api-client {:project-id "project-1"
                                  :client
                                   (simply.gcp.auth/default-credentials-client)})]
  (when-some [entity (lookup-entity client person-key)]
   (person entity))
```

**api**

Currently you can

* `lookup-entity` given a key
* `run-query` executes GQL against the DB
* `upsert-entity` or `upsert-entities` in a transaction
* `commit-mutations` for changing various entities in a transaction

You are limited by typical datastore constraints


## simply.gcp.pubsub.core

**topic**

Represents a pubsub topic that the system is allowed to push messages to. Takes a topic-key and a message encoder. Encoders are found at `simply.gcp.pubsub.encoding`.

**subscription**

Represents a pubsub subscription that the system is allowed to pull messages from. Takes a subscription-key, topic-key, message decoder, a message handler and the pull-interval in ms.

The message handler is a function that receives the decoded message as a parameter. Decoders are found at `simply.gcp.pubsub.encoding`.


**clients**

`production-client` and `emulator-client` wraps `simply.gcp.auth` clients to give access to Pubsub production and Pubsub emulator respectively.

**Pubsub Protocol**

Provides a protocol that allows for the easy consumption of the `pubsub` implementation.

**pubsub**

Implementation for `Pubsub Protocol`.
Takes a pubsub client, a collection of topics and a collection of subscriptions.

* **init** ensures that the required topics and subscriptions exist
* **subscribe-all** starts pulling messages from the topics subscribed to. Messages are decoded and sent to the message handler defined on the subscription
* **unsubscribe-all** stops pulling messages from the topics subscribed to
* **publish** publishes a message to a topic, encoding it with the supplied encoder

## simply.gcp.pubsub.encoding

Various encoders and decoders to use with pubsub topics and subscriptions.

## simply.gcp.pubsub.simply

Topics and subscriptions specific to simply.

## simply.gcp.pubsub.cqrs

`pubsub-cqrs-system` provides a `simply.cqrs/CqrsSystem` implementation that allows the use of GCP Pubsub for actions and event dispatching as well as other required notifications.

It takes the following parameters

* **client** a pubsub client. either `production-client` or `emulator-client`
* **service** the name of the service running the app
	* This name will set as the sender for events
	* It will be used in the name of the action topic `yourservice-required-actions`
* **events-topic** the topic key and encoder where events should be pushed
	* use `simply.gcp.pubsub.simply/events-topic` for simply systems
* **event-wrapper** an optional function that will wrap extra system data into all events that get dispatched
* **topics** a list of `simply.gcp.pubsub.core/topic` the system needs to push messages to. On startup the system will confirm if these topics exist
* **subscriptions** a list of `simply.gcp.pubsub.core/subscription` that the system will pull messages from

`ig/init-key :simply.gcp.pubsub.cqrs/pubsub-cqrs-system` is provided of use with Integrant.

**Example**

Below is a simple Integrant setup for initializing `pubsub-cqrs-system`.

See `simply.cqrs` for details on using handle-command.

```
;; system.edn

 :simply.gcp.pubsub.simply/events-topic
 {}

 :simply.gcp.pubsub.core/production-client
 {:project-id #dyn/prop [GCP_PROJECT_ID "dogmatix"]
  :client #ig/ref :simply.gcp.auth/default-credentials-client}

 :simply.gcp.pubsub.core/emulator-client
 {:project-id "dogmatix"
  :client #ig/ref :simply.gcp.auth/emulator-client
  :port 8321}

 :simply.gcp.pubsub.cqrs/pubsub-cqrs-system
 {:service "your-app"
  :client #ig/ref :simply.gcp.pubsub.core/production-client
  :events-topic #ig/ref :simply.gcp.pubsub.simply/events-topic
  :event-wrapper nil
  :topics []
  :subscriptions []}

 :your-app/handle-command
 {:cqrs-system #ig/ref :simply.gcp.pubsub.cqrs/pubsub-cqrs-system}

```


#Deploying a new version

see deploy.md
