;https://cloud.google.com/scheduler/docs/reference/rest/v1/projects.locations.jobs#Job
(ns simply.gcp.cloud-scheduler
  (:require [simply.errors :as e]
            [cheshire.core :as json]
            [clojure.spec.alpha :as s]
            [clojure.string :as string]
            [camel-snake-kebab.core :as csk]
            [clojure.walk :as walk]
            [base64-clj.core :as base64]))


;;;; SPEC
(s/def ::not-blank #(not (string/blank? %)))
(s/def ::duration ::not-blank)

(s/def ::retry-count #(<= 0 % 5))
(s/def ::max-retry-duration ::duration)
(s/def ::min-backoff-duration ::duration)
(s/def ::max-backoff-duration ::duration)
(s/def ::max-doublings number?)
(s/def ::retry-config (s/keys :req [::retry-count ::max-retry-duration
                                    ::min-backoff-duration ::max-backoff-duration
                                    ::max-doublings]))

(s/def ::topic-name ::not-blank)
(s/def ::data string?)
(s/def ::attributes (s/map-of keyword? string?))
(s/def ::pubsub-target (s/keys :req [::topic-name ::data ::attributes]))

(s/def ::app-engine-http-target (constantly false))
(s/def ::http-target (constantly false))

(s/def ::name ::not-blank)
(s/def ::description ::not-blank)
(s/def ::schedule ::not-blank)
(s/def ::time-zone ::not-blank)
(s/def ::attempt-deadline ::duration)
(s/def ::target (s/or ::pubsub-target ::pubsub-target
                      ::app-engine-http-target ::app-engine-http-target
                      ::http-target ::http-target))

(s/def ::job (s/and
              (s/keys :req [::name ::description ::schedule ::time-zone ::attempt-deadline ::retry-config]
                      :opt [::pubsub-target ::app-engine-http-target ::http-target])
              #(not (empty? (select-keys % [::pubsub-target ::app-engine-http-target ::http-target])))))


;;;; ENTITIES

(defn retry-config [& {:keys [retry-count max-retry-duration min-backoff-duration max-backoff-duration max-doublings]
                       :or {retry-count 3
                            max-retry-duration "0s"
                            min-backoff-duration "5s"
                            max-backoff-duration "3600s"
                            max-doublings 5}}]
  {::retry-count retry-count
   ::max-retry-duration max-retry-duration
   ::min-backoff-duration min-backoff-duration
   ::max-backoff-duration max-backoff-duration
   ::max-doublings max-doublings})


(defn pubsub-target
  [& {:keys [project-id topic data attributes]
      :or {attributes {}
           data ""}}]
  {::topic-name topic
   ::data (if (map? data) (pr-str data) data)
   ::attributes (->> attributes
                     (map (fn [[k v]]
                            [(name k) v]))
                     (into {}))})


(defn app-engine-http-target [& args]
  (e/throw-app-error "App engine http target not implemented"))


(defn http-target [& args]
  (e/throw-app-error "Http target not implemented"))


(defn job
  "see https://cloud.google.com/scheduler/docs/reference/rest/v1/projects.locations.jobs#Job"
  [& {:keys [name description schedule time-zone target attempt-deadline retry-config]
      :or {attempt-deadline "60s"
           retry-config (retry-config)}}]
  (let [t (s/conform ::target target)
        j
        (cond-> {::name name
                 ::description description
                 ::schedule schedule
                 ::time-zone time-zone
                 ::attempt-deadline attempt-deadline
                 ::retry-config retry-config}

          (not (= ::s/invalid t))
          (assoc (first t) (last t)))]
    (if (s/valid? ::job j)
      j
      (e/throw-app-error "Invalid Job" (s/explain-data ::job j)))))


;;;; API

(defn project->locations->list [& {:keys [page-token]}]
  {:method :get
   :url (str "/locations?pageSize=100"
             (if page-token (str "&pageToken=" page-token) ""))})


(defn project->locations->list->jobs->list [location & {:keys [page-token]}]
  {:method :get
   :url (str "/locations/" location "/jobs?pageSize=100" (if page-token (str "&pageToken=" page-token) ""))})


(defn project->locations->list->jobs->delete [location name]
  {:method :delete
   :url (str "/locations/" location "/jobs/" name)})


(defn project->locations->list->jobs->create [location job]
  (let [loc (str "/locations/" location "/jobs")]
    {:method :post
     :url loc
     :body (-> job
               (update ::name #(str loc "/" %)))}))


;;;; CLIENT
(defn- prep-pubsub-target [target project]
  (-> target
      (update ::topic-name #(str project "/topics/" %))
      (update ::data base64/encode)))


(defn- prep-body [body project]
  (let [prepped-body
        (cond-> body
          (contains? body ::name)
          (update ::name #(str project %))

          (contains? body ::pubsub-target)
          (update ::pubsub-target #(prep-pubsub-target % project)))]
    (->> prepped-body
         (walk/prewalk (fn [i]
                         (if (keyword? i)
                           (csk/->camelCase (name i))
                           i)))
         json/generate-string)))


(defn- prep-request [request project-id]
  (let [project (str "projects/" project-id)
        has-body? (contains? request :body)]
    (cond-> request
      true (update :url #(str "https://cloudscheduler.googleapis.com/v1/" project %))
      true (assoc :scope "https://www.googleapis.com/auth/cloud-platform")

      has-body? (update :body #(prep-body % project)))))


(defn ->client [project-id client]
  (fn [request]
    (-> request
        (prep-request project-id)
        client
        deref
        (update :body #(try (json/parse-string % keyword)
                            (catch Throwable _
                              %)))
        (select-keys [:body :status])
        (assoc :request request))))


(defn throw-response [response]
  (e/throw-app-error "Could not process scheduling"
                     response))


(defn wrap-throw-non-200 [client]
  (fn [request]
    (let [response (client request)]
      (when-not (= 200 (:status response))
        (throw-response response))
      response)))


(defn success-response? [response] (= 200 (:status response)))
(defn not-found-response? [response] (= 404 (:status response)))
(defn already-exists-response? [response] (= 409 (:status response)))


;;;; Helper Functions

(defn get-all-locations
  ([client]
   (let [response ((wrap-throw-non-200 client) (project->locations->list))
         {:keys [locations nextPageToken]} (:body response)
         locations (if-not (empty? nextPageToken)
                     (get-all-locations client nextPageToken locations)
                     locations)]
     (map :locationId locations)))
  ([client page-token results]
   (let [response ((wrap-throw-non-200 client) (project->locations->list :page-token page-token))
         {:keys [locations nextPageToken]} (:body response)
         locations (into results locations)]
     (if-not (empty? nextPageToken)
       (get-all-locations client nextPageToken locations)
       locations))))


(defn get-location-jobs
  ([client location]
   (let [response ((wrap-throw-non-200 client)
                   (project->locations->list->jobs->list location))
         {:keys [jobs nextPageToken]} (:body response)
         jobs (if-not (empty? nextPageToken)
                (get-location-jobs client location nextPageToken jobs)
                jobs)]
     jobs))
  ([client location page-token results]
   (let [response ((wrap-throw-non-200 client)
                   (project->locations->list->jobs->list location :page-token page-token))
         {:keys [jobs nextPageToken]} (:body response)
         jobs (into results jobs)]
     (if-not (empty? nextPageToken)
       (get-location-jobs client location nextPageToken jobs)
       jobs))))


(defn get-all-jobs-for-all-locations
  [client]
  (let [locations (get-all-locations client)]
    (->> locations
         (map (fn [location-id]
                (->> (get-location-jobs client location-id)
                     (map #(assoc % :location location-id)))))
         (reduce into))))


(comment
  (require 'simply.gcp.auth 'simply.gcp.keys)

  (simply.gcp.auth/reset-credentials!)

  "PROD"
  (let [f (->client "simply-prod" (simply.gcp.auth/keyfile-client (simply.gcp.keys/key-file :simply-prod)))]
    (f (project->locations->list)))


  (let [f (->client "simply-prod" (simply.gcp.auth/keyfile-client (simply.gcp.keys/key-file :simply-prod)))]
    (f (project->locations->list->jobs->list "us-east1")))

  "PREPROD"

  (let [f (->client "simply-pre-prod" (simply.gcp.auth/keyfile-client (simply.gcp.keys/key-file :simply-pre-prod)))]
    (f (project->locations->list)))


  (let [f (->client "simply-pre-prod" (simply.gcp.auth/keyfile-client (simply.gcp.keys/key-file :simply-pre-prod)))]
    (f (project->locations->list->jobs->list "us-central1")))


  (let [project-id "simply-pre-prod"
        client (->client project-id
                         (simply.gcp.auth/keyfile-client
                          (simply.gcp.keys/key-file :simply-pre-prod)))
        name "test-1"
        location "us-central1"
        job (job :name name
                 :description "test job"
                 :schedule "0 0 * * *"
                 :time-zone "Africa/Johannesburg"
                 :target (pubsub-target
                          :project-id project-id
                          :topic "events"
                          :data "{\"test\": 1}"))]
    #_(client (project->locations->list->jobs->create location job))

    ((wrap-throw-non-200 client) (project->locations->list->jobs->delete location name)))


  (get-all-locations (->client
                      "simply-pre-prod"
                      (simply.gcp.auth/keyfile-client (simply.gcp.keys/key-file :simply-pre-prod))))


  (get-location-jobs (->client
                      "simply-prod"
                      (simply.gcp.auth/keyfile-client (simply.gcp.keys/key-file :simply-prod)))
                     "us-east1")

  (get-all-jobs-for-all-locations (->client
                                   "simply-prod"
                                   (simply.gcp.auth/keyfile-client (simply.gcp.keys/key-file :simply-prod))))
  )
